
/* 
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: David Gruetter
   co-programmer: Benjamin Grauer
   
   Created by Dave: this file is actually quite similar to player.cc and so is 
   skybox.h similar to player.h
   With that said, things should be clear:)
   
   Edited:
   Bensch: more constructors, changeability, comments...
   Patrick: giving it the common orxonox style, not much to do... good work Dave!

*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_WORLD_ENTITY


#include "skybox.h"
#include "stdincl.h"

#include "material.h"
#include "vector.h"
#include "resource_manager.h"
#include "model.h"
//#include "world_entity.h"


using namespace std;

/**
   \brief Constructs a SkyBox and takes fileName as a map.
   \param fileName the file to take as input for the SkyBox
*/
SkyBox::SkyBox(char* fileName)
{
  this->setClassName("SkyBox");
  this->material = new Material*[6];
  for (int i = 0; i <6; i++) 
    {
      this->material[i] = new Material();
      this->material[i]->setIllum(3);
      this->material[i]->setDiffuse(0.0,0.0,0.0);
      this->material[i]->setSpecular(0.0,0.0,0.0);
      this->material[i]->setAmbient(2.0, 2.0, 2.0);
    }
  this->setMode(PNODE_MOVEMENT);

  this->setSize(1900.0);
}


/**
   \brief default destructor
*/
SkyBox::~SkyBox()
{
  PRINTF(5)("Deleting the SkyBox\n");
  
  for (int i = 0; i < 6; i++)
    delete this->material[i];
  delete []this->material;
}

/**
   \brief sets A set of textures when just giving a Name and an extension:

   usage: give this function an argument like
   setTexture("skybox", "jpg");
   and it will convert this to 
   setTextures("skybox_top.jpg", "skybox_bottom.jpg", "skybox_left.jpg",
               "skybox_right.jpg", "skybox_front.jpg", "skybox_back.jpg");
*/
void SkyBox::setTexture(const char* name, const char* extension)
{
  char* top    = new char[strlen(name)+strlen(extension)+ 6];
  char* bottom = new char[strlen(name)+strlen(extension)+ 9];
  char* left   = new char[strlen(name)+strlen(extension)+ 7];
  char* right  = new char[strlen(name)+strlen(extension)+ 8];
  char* front  = new char[strlen(name)+strlen(extension)+ 8];
  char* back   = new char[strlen(name)+strlen(extension)+ 7];

  sprintf(top, "%s_top.%s", name, extension);
  sprintf(bottom, "%s_bottom.%s", name, extension);
  sprintf(left, "%s_left.%s", name, extension);
  sprintf(right, "%s_right.%s", name, extension);
  sprintf(front, "%s_front.%s", name, extension);
  sprintf(back, "%s_back.%s", name, extension);
  
  this->setTextures(top, bottom, left, right, front, back);

  delete []top;
  delete []bottom;
  delete []left;
  delete []right;
  delete []front;
  delete []back;
}

/**
   \brief Defines which textures should be loaded onto the SkyBox.
   \param top the top texture.
   \param bottom the bottom texture.
   \param left the left texture.
   \param right the right texture.
   \param front the front texture.
   \param back the back texture.
*/
void SkyBox::setTextures(const char* top, const char* bottom, const char* left, const char* right, const char* front, const char* back)
{
  this->material[0]->setDiffuseMap(top);
  this->material[1]->setDiffuseMap(bottom);
  this->material[2]->setDiffuseMap(left);
  this->material[3]->setDiffuseMap(right);
  this->material[4]->setDiffuseMap(front);
  this->material[5]->setDiffuseMap(back);

  this->rebuild();
}

/**
   \param size The new size of the SkyBox
*/
void SkyBox::setSize(float size)
{
  this->size = size;

  this->rebuild();
}

/**
   \brief draws the SkyBox
*/
void SkyBox::draw()
{
  glPushMatrix();
  glMatrixMode(GL_MODELVIEW);
  Vector r = this->getAbsCoor();
  glTranslatef(r.x, r.y, r.z);

  this->model->draw();

  glPopMatrix();
}


/**
   \brief rebuilds the SkyBox
   
   this must be done, when changing the Size of the Skybox (runtime-efficency)
*/
void SkyBox::rebuild()
{
  if (this->model)
    delete model;
  model = new Model();

  model->addVertex (-0.5*size, -0.5*size, 0.5*size);  
  model->addVertex (0.5*size, -0.5*size, 0.5*size);
  model->addVertex (-0.5*size, 0.5*size, 0.5*size);
  model->addVertex (0.5*size, 0.5*size, 0.5*size);
  model->addVertex (-0.5*size, 0.5*size, -0.5*size);
  model->addVertex (0.5*size, 0.5*size, -0.5*size);
  model->addVertex (-0.5*size, -0.5*size, -0.5*size);
  model->addVertex (0.5*size, -0.5*size, -0.5*size);

  model->addVertexTexture (0.0, 0.0);
  model->addVertexTexture (1.0, 0.0);
  model->addVertexTexture (1.0, 1.0);
  model->addVertexTexture (0.0, 1.0);

  model->addVertexNormal (0.0, 0.0, 1.0);
  model->addVertexNormal (0.0, 1.0, 0.0);
  model->addVertexNormal (0.0, 0.0, -1.0);
  model->addVertexNormal (0.0, -1.0, 0.0);
  model->addVertexNormal (1.0, 0.0, 0.0);
  model->addVertexNormal (-1.0, 0.0, 0.0);

  model->addUseMtl(material[0]);
  model->addFace (4, 3, 3,2,4, 4,3,4, 6,4,4, 5,1,4); // top
  model->addUseMtl(material[1]);
  model->addFace (4, 3, 7,4,2, 8,1,2, 2,2,2, 1,3,2); // bottom
  model->addUseMtl(material[2]);
  model->addFace (4, 3, 1,1,3, 2,2,3, 4,3,3, 3,4,3); // left
  model->addUseMtl(material[3]);
  model->addFace (4, 3, 5,3,1, 6,4,1, 8,1,1, 7,2,1); // right
  model->addUseMtl(material[4]);
  model->addFace (4, 3, 2,1,6, 8,2,6, 6,3,6, 4,4,6); // front
  model->addUseMtl(material[5]);
  model->addFace (4, 3, 7,1,5, 1,2,5, 3,3,5, 5,4,5); // back
  
  model->finalize();
}
