/*! 
    \file glfont.h
    \brief Handles the display of glFonts.
*/

#ifndef _GLFONT_H
#define _GLFONT_H

#include "glincl.h"
#include "SDL_ttf.h"

// FORWARD DECLARATION 
template<class T> class tList;

/* some default values */
#define FONT_DEFAULT_SIZE       50                   //!< default size of the Text
#define FONT_DEFAULT_TEXT	"orxonox 1234567890" //!< some default text to display
#define FONT_DEFAULT_COLOR_R    256                  //!< the default red part (color) of the text
#define FONT_DEFAULT_COLOR_G    256                  //!< the default red green (color) of the text
#define FONT_DEFAULT_COLOR_B    256                  //!< the default red blue (color) of the text
#define FONT_NUM_COLORS         256                  //!< The number of colors.



//! A struct for handling glyphs
/**
   a Glyph is one letter of a certain font
*/
struct Glyph
{
  Uint16 character;              //!< The character
  int minX;                      //!< The minimum distance from the origin in X
  int maxX;                      //!< The maximum distance from the origin in X
  int minY;                      //!< The minimum distance from the origin in Y
  int maxY;                      //!< The maximum distance from the origin in Y
  int width;                     //!< The width of the Glyph
  int height;                    //!< The height of the Glyph
  int bearingX;                  //!< How much is right of the Origin
  int bearingY;                  //!< How much is above the Origin
  int advance;                   //!< How big a Glyph would be in monospace-mode
};

//! A Struct to handel Texture Coordinates for quads
struct TexCoord
{
  float minU;                      //!< The minimum U-Coordinate
  float maxU;                      //!< The maximum U-Coordinate
  float minV;                      //!< The minimum V-Coordinate
  float maxV;                      //!< The maximum V-Coordinate
};

//! A class to handle a Font
class GLFont
{
 public:
  GLFont(const char* fontFile);
  virtual ~GLFont();

  static void enableFonts(void);
  static void disableFonts(void);

  bool setFont(const char* fontFile);
  void setText(const char* text);

  void setStyle(char* renderStyle);
  void setSize(unsigned int fontSize);
  void setColor(Uint8 r, Uint8 g, Uint8 b);
  void setPosition(int x, int y);

  void createTexture(void);
  
  virtual void draw(void);

 private:
  // information about the Font
  TTF_Font* font;                  //!< The font we use for this.
  char* fontFile;                  //!< The fontfile from whitch the font was loaded.
  unsigned int fontSize;           //!< The size of the font in pixels. each Font has one size.

  //! Represents one textElement.
  struct Text
  {
    char* text;                      //!< The text to display
    SDL_Color color;                 //!< The color of the font.
    // placement in openGL
    GLuint texture;                  //!< A GL-texture to hold the text
    TexCoord texCoord;               //!< Texture-coordinates \todo fix this to have a struct
    SDL_Rect textPosSize;            //!< An SDL-Rectangle representing the position and size of the Text on the screen.
    int renderStyle;                 //!< The Renderstyle
  };
  tList<Text>* textList;
  Text* currentText;

  bool init(const char* fontFile, unsigned int fontSize = FONT_DEFAULT_SIZE);
  int getMaxHeight(void);
  int getMaxAscent(void);
  int getMaxDescent(void);
  Glyph getGlyphMetrics(Uint16 character);

  static bool ttfInitialized;

  void enter2DMode(void);
  void leave2DMode(void);

  static bool checkVersion(void);

  GLuint loadTexture(SDL_Surface* surface, TexCoord* texCoord);

  static int powerOfTwo(int input);

  void debug(void);

};

#endif /* _GLFONT_H */
