/* 
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...

   ***
   * This file is extended to the needs of the orxonox-project.         *
   * the Copyright of the original file is below this copyright         *
   *                                                                  ***

   for some fonts and licenses visit: =http://www.dafont.com/en/font.php=

   !! IMPORTANT !! When using ttf fonts clear the license issues prior to
   adding them to orxonox. This is really important, because we do not 
   want to defend anyone.
*/

/*
  glfont:  An example of using the SDL_ttf library with OpenGL.
  Copyright (C) 1997-2004 Sam Lantinga
  
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.
  
  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
  
  The SDL_GL_* functions in this file are available in the public domain.
  
  Sam Lantinga
  slouken@libsdl.org
*/

#include "glfont.h"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "debug.h"

/**
   \brief constructs a Font
   \param fontFile the File to load the font from
*/
GLFont::GLFont(const char* fontFile)
{
  this->init(fontFile);
}

/**
   \brief destructs a font
*/
GLFont::~GLFont(void)
{
  delete this->currentText;

  if (this->font)
    TTF_CloseFont(this->font);
}

/**
   \brief function to enable TTF_Fonts
*/
void GLFont::enableFonts(void)
{
  if (!GLFont::ttfInitialized)
    {
      if(TTF_Init()==-1)
	PRINTF(1)("TTF_Init: %s\n", TTF_GetError());

      GLFont::checkVersion();
      GLFont::ttfInitialized = true;
    }
  else
    PRINTF(4)("Fonts already initialized\n");
      
}

/**
   \brief function to disable TTF_fonts
*/
void GLFont::disableFonts(void)
{
  if (GLFont::ttfInitialized)
    {
      TTF_Quit();
      GLFont::ttfInitialized = false;
    }
  else
    PRINTF(4)("Fonts were not initialized.\n");
}

//! A simple variable for checking if ttf was initialized
bool GLFont::ttfInitialized = false;

/**
   \brief initializes a new Font
   \param fontFile The file to load a Font from
   \param fontSize the Size in pixels of the Font
*/
bool GLFont::init(const char* fontFile, unsigned int fontSize)
{
  if (!GLFont::ttfInitialized)
    GLFont::enableFonts();

  // setting default values.
  this->font = NULL;
  this->fontFile = NULL;

  this->currentText = new Text;

  this->currentText->text = NULL;
  this->currentText->texture = 0;
  
  this->setSize(fontSize);
  
  this->currentText->renderStyle = TTF_STYLE_NORMAL;

  this->setFont(fontFile);

  this->setColor(0, 255, 0);

  this->setText(FONT_DEFAULT_TEXT);

  this->createTexture();
}

/**
   \brief sets The Font.
   \param fontFile The file containing the font.
   \returns true if loaded, false if something went wrong, or if a font was loaded before.
*/
bool GLFont::setFont(const char* fontFile)
{
  if (!this->fontFile)
    {
      this->fontFile = new char[strlen(fontFile)+1];
      strcpy(this->fontFile, fontFile);
      
      this->font = TTF_OpenFont(this->fontFile, this->fontSize);
      if(!this->font) 
	{
	  PRINTF(1)("TTF_OpenFont: %s\n", TTF_GetError());
	  return false;
      }
      return true;
    }
  else
    {
      PRINTF(2)("Font already initialized, unable to change it now.\n");
      return false;
    }
}

/**
   \brief Sets a new Text to the font
   \param text the new text to set
*/
void GLFont::setText(const char* text)
{
  if (this->currentText->text)
    delete []this->currentText->text;
  this->currentText->text = new char[strlen(text)+1];
  strcpy(this->currentText->text, text);
}

/**
   \brief sets a specific renderStyle
   \param renderStyle the Style to render: a char-array containing:
   i: italic, b: bold, u, underline
*/
void GLFont::setStyle(char* renderStyle)
{
  this->currentText->renderStyle = TTF_STYLE_NORMAL;
  
  for (int i = 0; i < strlen(renderStyle); i++)
    if (strncmp(renderStyle+i, "b", 1) == 0) 
      this->currentText->renderStyle |= TTF_STYLE_BOLD;
    else if (strncmp(renderStyle+i, "i", 1) == 0)
      this->currentText->renderStyle |= TTF_STYLE_ITALIC;
    else if (strncmp(renderStyle+i, "u", 1) == 0) 
      this->currentText->renderStyle |= TTF_STYLE_UNDERLINE;

  if (this->font)
    TTF_SetFontStyle(this->font, this->currentText->renderStyle);
  else
    PRINTF(2)("Font was not initialized, please do so before setting the Font-Style.\n");
}

/**
   \brief Sets a new Size to the font
   \param fontSize The new Size in pixels.
*/
void GLFont::setSize(unsigned int fontSize)
{
  this->fontSize = fontSize;
}

/**
   \brief sets a new color to the font
   \param r Red
   \param g Green
   \param b Blue
*/
void GLFont::setColor(Uint8 r, Uint8 g, Uint8 b)
{
  this->currentText->color.r = r;
  this->currentText->color.g = g;
  this->currentText->color.b = b;
}

/**
   \brief sets a Position.
   \param x the x-position in pixels from the left border
   \param y the y-position in pixels from the top border
*/
void GLFont::setPosition(int x, int y)
{
  this->currentText->textPosSize.x = x;
  this->currentText->textPosSize.y = y;
}

/**
   \brief draws the Font
   \todo FIX this is to slow/static
*/
void GLFont::draw(void)
{
  this->enter2DMode();

  glBindTexture(GL_TEXTURE_2D, this->currentText->texture);
  glEnable(GL_TEXTURE_2D);
  glBegin(GL_QUADS);

  glTexCoord2f(this->currentText->texCoord.minU, this->currentText->texCoord.minV);
  glVertex2i(20,   20  );

  glTexCoord2f(this->currentText->texCoord.maxU, this->currentText->texCoord.minV);
  glVertex2i(20+this->currentText->textPosSize.w, 20  );

  glTexCoord2f(this->currentText->texCoord.maxU, this->currentText->texCoord.maxV);
  glVertex2i(20+this->currentText->textPosSize.w, 20+this->currentText->textPosSize.h);

  glTexCoord2f(this->currentText->texCoord.minU, this->currentText->texCoord.maxV);
  glVertex2i(20, 20+this->currentText->textPosSize.h);

  glEnd();

  this->leave2DMode();
}

/**
   \brief creates a texture out of the given parameters

   this has to be called every time by the user, to if changes were made.
*/
void GLFont::createTexture(void)
{
  SDL_Surface* tmpSurf;
  if (this->currentText->texture)
    glDeleteTextures(1, &this->currentText->texture);
  tmpSurf = TTF_RenderText_Blended(this->font,
				   this->currentText->text,
				   this->currentText->color);
  if (tmpSurf)
    this->currentText->texture = loadTexture(tmpSurf, &this->currentText->texCoord);

  this->currentText->textPosSize.w = tmpSurf->w;
  this->currentText->textPosSize.h = tmpSurf->h;
  SDL_FreeSurface(tmpSurf);
}


/**
   \returns the maximum height of the Font, if the font was initialized, 0 otherwise
*/
int GLFont::getMaxHeight(void)
{
  if (this->font)
    return TTF_FontHeight(this->font);
  else
    return 0;
}

/**
   \returns the maximum ascent of the Font, if the font was initialized, 0 otherwise

   the ascent is the pixels of the font above the baseline
*/
int GLFont::getMaxAscent(void)
{
  if (this->font)
    return TTF_FontAscent(this->font);
  else
    return 0;
}

/**
   \returns the maximum descent of the Font, if the font was initialized, 0 otherwise

   the descent is the pixels of the font below the baseline
*/
int GLFont::getMaxDescent(void)
{
  if (this->font)
    return TTF_FontDescent(this->font);
  else
    return 0;
}

/**
   \param character The character to get info about.
   \returns a Glyph struct of a character.

   This only works for horizontal fonts. see 
   http://freetype.sourceforge.net/freetype2/docs/tutorial/step2.html
   for more info about vertical Fonts
*/
Glyph GLFont::getGlyphMetrics(Uint16 character)
{
  Glyph rg;
  rg.character = character;
  TTF_GlyphMetrics(this->font, rg.character,
		   &rg.minX, &rg.maxX,
		   &rg.minY, &rg.maxY,
		   &rg.advance);
  rg.height = rg.maxY - rg.minY;
  rg.width = rg.maxX - rg.minX;
  rg.bearingX = (rg.advance - rg.width) / 2;
  rg.bearingY = rg.maxY;
  return rg;
}

/**
   \brief entering 2D Mode
   
   this is a GL-Projection-mode, that is orthogonal, for placing the font in fron of everything else
*/
void GLFont::enter2DMode(void)
{
  SDL_Surface *screen = SDL_GetVideoSurface();
  
  /* Note, there may be other things you need to change,
     depending on how you have your OpenGL state set up.
  */
  glPushAttrib(GL_ENABLE_BIT);
  glDisable(GL_DEPTH_TEST);
  glDisable(GL_CULL_FACE);
  glDisable(GL_LIGHTING);  // will be set back when leaving 2D-mode
  glEnable(GL_TEXTURE_2D);

  /* This allows alpha blending of 2D textures with the scene */
  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  
  glViewport(0, 0, screen->w, screen->h);
  
  glMatrixMode(GL_PROJECTION);
  glPushMatrix();
  glLoadIdentity();
  
  glOrtho(0.0, (GLdouble)screen->w, (GLdouble)screen->h, 0.0, 0.0, 1.0);
  
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  glLoadIdentity();
  
  glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
}

/**
   \brief leaves the 2DMode again also \see GLFont::enter2DMode(void)
*/
void GLFont::leave2DMode(void)
{
	glMatrixMode(GL_MODELVIEW);
	glPopMatrix();

	glMatrixMode(GL_PROJECTION);
	glPopMatrix();

	glPopAttrib();
}

/**
   \brief Loads a Font from an SDL_surface into a texture.
   \param surface The surface to make the texture of
   \param texCoord The texture coordinates of the 4 corners of the texture
   \returns the ID of the texture
*/
GLuint GLFont::loadTexture(SDL_Surface *surface, TexCoord* texCoord)
{
  GLuint texture;
  int w, h;
  SDL_Surface *image;
  SDL_Rect area;
  Uint32 saved_flags;
  Uint8  saved_alpha;
  
  /* Use the surface width and height expanded to powers of 2 */
  w = powerOfTwo(surface->w);
  h = powerOfTwo(surface->h);
  texCoord->minU = 0.0f;
  texCoord->minV = 0.0f;
  texCoord->maxU = (GLfloat)surface->w / w;
  texCoord->maxV = (GLfloat)surface->h / h;
  
  image = SDL_CreateRGBSurface(
			       SDL_SWSURFACE,
			       w, h,
			       32,
#if SDL_BYTEORDER == SDL_LIL_ENDIAN /* OpenGL RGBA masks */
			       0x000000FF, 
			       0x0000FF00, 
			       0x00FF0000, 
			       0xFF000000
#else
			       0xFF000000,
			       0x00FF0000, 
			       0x0000FF00, 
			       0x000000FF
#endif
			       );
  if ( image == NULL ) {
    return 0;
  }
  
  /* Save the alpha blending attributes */
  saved_flags = surface->flags&(SDL_SRCALPHA|SDL_RLEACCELOK);
  saved_alpha = surface->format->alpha;
  if ( (saved_flags & SDL_SRCALPHA) == SDL_SRCALPHA ) {
    SDL_SetAlpha(surface, 0, 0);
  }
  
  /* Copy the surface into the GL texture image */
  area.x = 0;
  area.y = 0;
  area.w = surface->w;
  area.h = surface->h;
  SDL_BlitSurface(surface, &area, image, &area);
  
  /* Restore the alpha blending attributes */
  if ( (saved_flags & SDL_SRCALPHA) == SDL_SRCALPHA ) {
    SDL_SetAlpha(surface, saved_flags, saved_alpha);
  }
  
  /* Create an OpenGL texture for the image */
  glGenTextures(1, &texture);
  glBindTexture(GL_TEXTURE_2D, texture);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
  glTexImage2D(GL_TEXTURE_2D,
	       0,
	       GL_RGBA,
	       w, h,
	       0,
	       GL_RGBA,
	       GL_UNSIGNED_BYTE,
	       image->pixels);
  SDL_FreeSurface(image); /* No longer needed */
  
  return texture;
}

/**
   \brief Quick utility function for texture creation 
   \param input an integer
   \returns the next bigger 2^n-integer than input
*/
int GLFont::powerOfTwo(int input)
{
  int value = 1;
  
  while ( value < input ) {
    value <<= 1;
  }
  return value;
}


/**
   \brief checks if the compiled version and the local version of SDL_ttf match.
   \returns true if match, false otherwise
*/
bool GLFont::checkVersion(void)
{
  SDL_version compile_version;
  SDL_version link_version;
  TTF_VERSION(&compile_version);
  link_version = *TTF_Linked_Version();

  if (compile_version.major == link_version.major &&
      compile_version.minor == link_version.minor &&
      compile_version.patch == link_version.patch)
    {
      return true;
    }
  else
    {
      PRINTF(2)("compiled with SDL_ttf version: %d.%d.%d\n", 
		compile_version.major,
		compile_version.minor,
		compile_version.patch);
      
      PRINTF(2)("running with SDL_ttf version: %d.%d.%d\n", 
		link_version.major,
		link_version.minor,
		link_version.patch);
      return false;
    }
}



/**
   \brief a simple function to get some interesting information about this class
*/
void GLFont::debug(void)
{

  // print the loaded font's style
  int style;
  style=TTF_GetFontStyle(this->font);
  PRINTF(0)("The font style is:");
  if(style==TTF_STYLE_NORMAL)
    PRINTF(0)(" normal");
  else {
    if(style&TTF_STYLE_BOLD)
      PRINTF(0)(" bold");
    if(style&TTF_STYLE_ITALIC)
      PRINTF(0)(" italic");
    if(style&TTF_STYLE_UNDERLINE)
      PRINTF(0)(" underline");
  }
  PRINTF(0)("\n");


}
