/* 
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...

   ***
   * This file is extended to the needs of the orxonox-project.         *
   * the Copyright of the original file is below this copyright         *
   *                                                                  ***

   for some fonts and licenses visit: =http://www.dafont.com/en/font.php=
*/

/*
    glfont:  An example of using the SDL_ttf library with OpenGL.
    Copyright (C) 1997-2004 Sam Lantinga

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    The SDL_GL_* functions in this file are available in the public domain.

    Sam Lantinga
    slouken@libsdl.org
*/

#include "glfont.h"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "debug.h"

GLFont::GLFont(const char* fontFile, unsigned int fontSize)
{
  this->init(fontFile, fontSize);
}

GLFont::~GLFont(void)
{
  if (this->font)
    TTF_CloseFont(this->font);
}

/**
   \function to enable TTF_Fonts
*/
void GLFont::enableFonts(void)
{
  if (!GLFont::ttfInitialized)
    {
      if(TTF_Init()==-1) {
	PRINTF(1)("TTF_Init: %s\n", TTF_GetError());
	exit(2);
      }
      GLFont::checkVersion();
      GLFont::ttfInitialized = true;
    }
  else
    PRINTF(4)("Fonts already initialized\n");
      
}

/**
   \brief function to disable TTF_fonts
*/
void GLFont::disableFonts(void)
{
  if (GLFont::ttfInitialized)
    {
      TTF_Quit();
      GLFont::ttfInitialized = false;
    }
  else
    PRINTF(4)("Fonts were not initialized.\n");
}

bool GLFont::ttfInitialized = false;


bool GLFont::init(const char* fontFile, unsigned int fontSize)
{
  if (!GLFont::ttfInitialized)
    GLFont::enableFonts();

  // setting default values.
  this->font = NULL;
  this->fontFile = NULL;

  
  this->setSize(fontSize);
  
  this->renderStyle = TTF_STYLE_NORMAL;

  this->setFont(fontFile);

  this->setText( FONT_DEFAULT_TEXT);

}


bool GLFont::setFont(const char* fontFile)
{
  if (!this->fontFile)
    {
      this->fontFile = new char[strlen(fontFile)+1];
      strcpy(this->fontFile, fontFile);
      
      this->font = TTF_OpenFont(this->fontFile, this->fontSize);
      if(!this->font) 
	{
	  PRINTF(1)("TTF_OpenFont: %s\n", TTF_GetError());
	  // handle error
      }
    }
  else
    PRINTF(2)("Font already initialized, unable to change it now.\n");
}


void GLFont::setText(const char* text)
{
  this->text = new char[strlen(text)+1];
  strcpy(this->text, text);

  this->createTexture();
}

/**
   \brief sets a specific renderStyle
   \param renderStyle the Style to render: a char-array containing:
   i: italic, b: bold, u, underline
*/
void GLFont::setStyle(char* renderStyle)
{
  this->renderStyle = TTF_STYLE_NORMAL;
  
  for (int i = 0; i < strlen(renderStyle); i++)
    if (strncmp(renderStyle+i, "b", 1) == 0) 
      this->renderStyle |= TTF_STYLE_BOLD;
    else if (strncmp(renderStyle+i, "i", 1) == 0)
      this->renderStyle |= TTF_STYLE_ITALIC;
    else if (strncmp(renderStyle+i, "u", 1) == 0) 
      this->renderStyle |= TTF_STYLE_UNDERLINE;

  if (this->font)
    TTF_SetFontStyle(this->font, this->renderStyle);
  else
    PRINTF(2)("Font was not initialized, please do so before setting the Font-Style.\n");
}

void GLFont::setSize(unsigned int fontSize)
{
  this->fontSize = fontSize;
}

void GLFont::setColor(Uint8 r, Uint8 g, Uint8 b)
{
  this->color.r = r;
  this->color.g = g;
  this->color.b = b;
}


void GLFont::setPosition(int x, int y)
{
  this->positionX = x;
  this->positionY = y;
}

void GLFont::draw(void)
{
  this->enter2DMode();
  glBindTexture(GL_TEXTURE_2D, texture);
  glEnable(GL_TEXTURE_2D);
  glBegin(GL_QUADS);
  glTexCoord2f(0, 0); glVertex2i(20,   20  );
  glTexCoord2f(1, 0); glVertex2i(200, 20  );
  glTexCoord2f(1, 1); glVertex2i(200,   80);
  glTexCoord2f(0, 1); glVertex2i(20, 80);
  glEnd();
  this->leave2DMode();

}



void GLFont::createTexture(void)
{
  GLfloat texcoord[4];
  SDL_Surface* tmpSurf = TTF_RenderText_Blended(this->font, this->text, this->color);
  if (tmpSurf)
    this->texture = loadTexture(tmpSurf, texcoord);
}


/**
   \returns the maximum height of the Font, if the font was initialized, 0 otherwise
*/
int GLFont::getMaxHeight(void)
{
  if (this->font)
    return TTF_FontHeight(this->font);
  else
    return 0;
}

/**
   \returns the maximum ascent of the Font, if the font was initialized, 0 otherwise

   the ascent is the pixels of the font above the baseline
*/
int GLFont::getMaxAscent(void)
{
  if (this->font)
    return TTF_FontAscent(this->font);
  else
    return 0;
}

/**
   \returns the maximum descent of the Font, if the font was initialized, 0 otherwise

   the descent is the pixels of the font below the baseline
*/
int GLFont::getMaxDescent(void)
{
  if (this->font)
    return TTF_FontDescent(this->font);
  else
    return 0;
}

/**
   \param character The character to get info about.
   \returns a Glyph struct of a character.

   This only works for horizontal fonts. see 
   http://freetype.sourceforge.net/freetype2/docs/tutorial/step2.html
   for more info about vertical Fonts
*/
glyph GLFont::getGlyphMetrics(Uint16 character)
{
  glyph rg;
  rg.character = character;
  TTF_GlyphMetrics(this->font, rg.character,
		   &rg.minX, &rg.maxX,
		   &rg.minY, &rg.maxY,
		   &rg.advance);
  rg.height = rg.maxY - rg.minY;
  rg.width = rg.maxX - rg.minX;
  rg.bearingX = (rg.advance - rg.width) / 2;
  rg.bearingY = rg.maxY;
  return rg;
}

void GLFont::renderText(void)
{

}



void GLFont::enter2DMode()
{
  SDL_Surface *screen = SDL_GetVideoSurface();
  
  /* Note, there may be other things you need to change,
     depending on how you have your OpenGL state set up.
  */
  glPushAttrib(GL_ENABLE_BIT);
  glDisable(GL_DEPTH_TEST);
  glDisable(GL_CULL_FACE);
  glEnable(GL_TEXTURE_2D);
  
  /* This allows alpha blending of 2D textures with the scene */
  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  
  glViewport(0, 0, screen->w, screen->h);
  
  glMatrixMode(GL_PROJECTION);
  glPushMatrix();
  glLoadIdentity();
  
  glOrtho(0.0, (GLdouble)screen->w, (GLdouble)screen->h, 0.0, 0.0, 1.0);
  
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  glLoadIdentity();
  
  glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
}


void GLFont::leave2DMode()
{
	glMatrixMode(GL_MODELVIEW);
	glPopMatrix();

	glMatrixMode(GL_PROJECTION);
	glPopMatrix();

	glPopAttrib();
}


GLuint GLFont::loadTexture(SDL_Surface *surface, GLfloat *texcoord)
{
  GLuint texture;
  int w, h;
  SDL_Surface *image;
  SDL_Rect area;
  Uint32 saved_flags;
  Uint8  saved_alpha;
  
  /* Use the surface width and height expanded to powers of 2 */
  w = powerOfTwo(surface->w);
  h = powerOfTwo(surface->h);
  texcoord[0] = 0.0f;			/* Min X */
  texcoord[1] = 0.0f;			/* Min Y */
  texcoord[2] = (GLfloat)surface->w / w;	/* Max X */
  texcoord[3] = (GLfloat)surface->h / h;	/* Max Y */
  
  image = SDL_CreateRGBSurface(
			       SDL_SWSURFACE,
			       w, h,
			       32,
#if SDL_BYTEORDER == SDL_LIL_ENDIAN /* OpenGL RGBA masks */
			       0x000000FF, 
			       0x0000FF00, 
			       0x00FF0000, 
			       0xFF000000
#else
			       0xFF000000,
			       0x00FF0000, 
			       0x0000FF00, 
			       0x000000FF
#endif
			       );
  if ( image == NULL ) {
    return 0;
  }
  
  /* Save the alpha blending attributes */
  saved_flags = surface->flags&(SDL_SRCALPHA|SDL_RLEACCELOK);
  saved_alpha = surface->format->alpha;
  if ( (saved_flags & SDL_SRCALPHA) == SDL_SRCALPHA ) {
    SDL_SetAlpha(surface, 0, 0);
  }
  
  /* Copy the surface into the GL texture image */
  area.x = 0;
  area.y = 0;
  area.w = surface->w;
  area.h = surface->h;
  SDL_BlitSurface(surface, &area, image, &area);
  
  /* Restore the alpha blending attributes */
  if ( (saved_flags & SDL_SRCALPHA) == SDL_SRCALPHA ) {
    SDL_SetAlpha(surface, saved_flags, saved_alpha);
  }
  
  /* Create an OpenGL texture for the image */
  glGenTextures(1, &texture);
  glBindTexture(GL_TEXTURE_2D, texture);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
  glTexImage2D(GL_TEXTURE_2D,
	       0,
	       GL_RGBA,
	       w, h,
	       0,
	       GL_RGBA,
	       GL_UNSIGNED_BYTE,
	       image->pixels);
  SDL_FreeSurface(image); /* No longer needed */
  
  return texture;
}


bool GLFont::checkVersion(void)
{
  SDL_version compile_version;
  SDL_version link_version;
  TTF_VERSION(&compile_version);
  link_version = *TTF_Linked_Version();

  if (compile_version.major == link_version.major &&
      compile_version.minor == link_version.minor &&
      compile_version.patch == link_version.patch)
    {
      return true;
    }
  else
    {
      PRINTF(2)("compiled with SDL_ttf version: %d.%d.%d\n", 
		compile_version.major,
		compile_version.minor,
		compile_version.patch);
      
      PRINTF(2)("running with SDL_ttf version: %d.%d.%d\n", 
		link_version.major,
		link_version.minor,
		link_version.patch);
      return false;
    }
}



/* Quick utility function for texture creation */
int GLFont::powerOfTwo(int input)
{
  int value = 1;
  
  while ( value < input ) {
    value <<= 1;
  }
  return value;
}



void GLFont::debug(void)
{

  // print the loaded font's style
  int style;
  style=TTF_GetFontStyle(this->font);
  PRINTF(0)("The font style is:");
  if(style==TTF_STYLE_NORMAL)
    PRINTF(0)(" normal");
  else {
    if(style&TTF_STYLE_BOLD)
      PRINTF(0)(" bold");
    if(style&TTF_STYLE_ITALIC)
      PRINTF(0)(" italic");
    if(style&TTF_STYLE_UNDERLINE)
      PRINTF(0)(" underline");
  }
  PRINTF(0)("\n");


}
