

/* 
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific
   main-programmer: Patrick Boenzli
   co-programmer: 
*/


#include "primitive.h"

#include "stdincl.h"
#include "world_entity.h"
#include "model.h"
#include "vector.h"

using namespace std;



/**
   \brief standard constructor
*/
Primitive::Primitive (pForm form) : WorldEntity()
{
  this->form = form;
  this->size = 0.2f;
  this->segments = 6;
  this->object = gluNewQuadric();
  
  gluQuadricNormals(this->object, GLU_SMOOTH); 
  gluQuadricTexture(this->object, GL_TRUE);
  
  this->material = new Material("Sphere");
  this->material->setIllum(3);
  this->material->setAmbient(1, .5, 1);
}


/**
   \brief standard deconstructor
*/
Primitive::~Primitive () 
{
  delete this->material;
  free(this->object);
}
 

/**
   \brief sets the form of the primitive body
   \param form (enumeration) from pForm
*/
void Primitive::setForm(pForm form)
{
  this->form = form;
}


/**
   \brief sets the size of the body
   \param size from [0..1]
*/
void Primitive::setSize(float size)
{
  if( size < 0 || size > 1)
    {
      PRINTF(1)("size must be value between 0 and 1 [0..1]. resetting it to 1\n");
      size = 1.0;
    }
  this->size = size;
}


/**
   \brief set number of segments
   \param segment count

   the more segments the body has, the smoother it seems. This function makes only
   sense with speres and cylinders, cubes have constant number of segments :)
*/
void Primitive::setSegments(int segments)
{
  if( segments <= 0)
    {
      PRINTF(1)("number of segments must be more than 0! not setting segment number\n");
    }
  this->segments = segments;
}

/**
   \brief stets the material
   \param material to be set
*/
void Primitive::setMaterial(Material* material)
{
  if( material == NULL)
    {
      PRINTF(1)("material reference is NULL, returning without modifications\n");
      return;
    }
  this->material = material;
}


/**
   \brief sets the texture name
   \param path to the texture

   eg: name could have following value: "../data/pictures/load_screen.jpg"
*/
void Primitive::setTexture(char* name)
{
  if( name == NULL )
    {
      PRINTF(1)("texture char reference is NULL, returning without modifications\n");
      return;
    }
  this->textureName = name;
  this->material->setDiffuseMap( name);
}


/**
   \brief called, when the object gets hit
   \param other object, that hits it
   \param location of impact
*/
void Primitive::hit (WorldEntity* weapon, Vector* loc) {}


/**
   \brief object collides
*/
void Primitive::collide (WorldEntity* other,  Uint32 ownhitflags, Uint32 otherhitflags) {}


/**
   \brief tick singal put everything here, that is timedependent
   \param time in sec
*/
inline void Primitive::tick (float time) 
{
  // Vector v(0.0, 0.0, 1.0);
  // Quaternion q(10.0, v);
  // this->setRelDir(&(this->relDirection * q));
}

/**
   \brief drawing function of the object
*/
inline void Primitive::draw () 
{
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();

  float matrix[4][4];
  glTranslatef (this->getAbsCoor ().x, this->getAbsCoor ().y, this->getAbsCoor ().z);
  //rotate
  //this->getAbsDir().matrix (matrix);
  //glMultMatrixf((float*)matrix);
  this->material->select();
  
  switch( this->form)
    {
    case P_SPHERE:
      gluSphere(this->object, this->size, this->segments, this->segments);
      break;
    case P_CUBE:
      this->glDrawCube();
      break;
    case P_CYLINDER:
      gluCylinder(this->object, this->size, this->size, this->size * 2, this->segments, this->segments);
      break;
    case P_DISK:
      gluDisk(this->object, this->size, 2, this->segments, this->segments);
      break;
    default:
      break;
    }

  glPopMatrix();
}


void Primitive::glDrawCube()				
{
  glBegin(GL_QUADS);			
  glNormal3f( 0.0f, 0.0f, 1.0f);	
  glTexCoord2f(0.0f, 0.0f); glVertex3f(-1.0f, -1.0f,  1.0f);
  glTexCoord2f(1.0f, 0.0f); glVertex3f( 1.0f, -1.0f,  1.0f);
  glTexCoord2f(1.0f, 1.0f); glVertex3f( 1.0f,  1.0f,  1.0f);
  glTexCoord2f(0.0f, 1.0f); glVertex3f(-1.0f,  1.0f,  1.0f);

  glNormal3f( 0.0f, 0.0f,-1.0f);	
  glTexCoord2f(1.0f, 0.0f); glVertex3f(-1.0f, -1.0f, -1.0f);
  glTexCoord2f(1.0f, 1.0f); glVertex3f(-1.0f,  1.0f, -1.0f);
  glTexCoord2f(0.0f, 1.0f); glVertex3f( 1.0f,  1.0f, -1.0f);
  glTexCoord2f(0.0f, 0.0f); glVertex3f( 1.0f, -1.0f, -1.0f);

  glNormal3f( 0.0f, 1.0f, 0.0f);	
  glTexCoord2f(0.0f, 1.0f); glVertex3f(-1.0f,  1.0f, -1.0f);
  glTexCoord2f(0.0f, 0.0f); glVertex3f(-1.0f,  1.0f,  1.0f);
  glTexCoord2f(1.0f, 0.0f); glVertex3f( 1.0f,  1.0f,  1.0f);
  glTexCoord2f(1.0f, 1.0f); glVertex3f( 1.0f,  1.0f, -1.0f);

  glNormal3f( 0.0f,-1.0f, 0.0f);
  glTexCoord2f(1.0f, 1.0f); glVertex3f(-1.0f, -1.0f, -1.0f);
  glTexCoord2f(0.0f, 1.0f); glVertex3f( 1.0f, -1.0f, -1.0f);
  glTexCoord2f(0.0f, 0.0f); glVertex3f( 1.0f, -1.0f,  1.0f);
  glTexCoord2f(1.0f, 0.0f); glVertex3f(-1.0f, -1.0f,  1.0f);

  glNormal3f( 1.0f, 0.0f, 0.0f);
  glTexCoord2f(1.0f, 0.0f); glVertex3f( 1.0f, -1.0f, -1.0f);
  glTexCoord2f(1.0f, 1.0f); glVertex3f( 1.0f,  1.0f, -1.0f);
  glTexCoord2f(0.0f, 1.0f); glVertex3f( 1.0f,  1.0f,  1.0f);
  glTexCoord2f(0.0f, 0.0f); glVertex3f( 1.0f, -1.0f,  1.0f);

  glNormal3f(-1.0f, 0.0f, 0.0f);
  glTexCoord2f(0.0f, 0.0f); glVertex3f(-1.0f, -1.0f, -1.0f);
  glTexCoord2f(1.0f, 0.0f); glVertex3f(-1.0f, -1.0f,  1.0f);
  glTexCoord2f(1.0f, 1.0f); glVertex3f(-1.0f,  1.0f,  1.0f);
  glTexCoord2f(0.0f, 1.0f); glVertex3f(-1.0f,  1.0f, -1.0f);
  glEnd();
}
