/*! 
    \file resource_manager.h
    \brief The Resource Manager checks if a file/resource is loaded.

    If a file/resource was already loaded the resourceManager will 
    return a void pointer to the desired resource.
    Otherwise it will instruct the coresponding resource-loader to load,
    and receive a pointer to it.
*/

#ifndef _RESOURCE_MANAGER_H
#define _RESOURCE_MANAGER_H

#include "base_object.h"

// FORWARD DEFINITION \\
//template<class T> class tList;
#include "list.h"                //! \todo do this by forward definition (ask Patrick)

//! An eumerator for different fileTypes the resourceManager supports \todo WAV, MP3, OGG support
enum ResourceType {OBJ, PRIM, WAV, MP3, OGG, IMAGE};
//! An enumerator for different UNLOAD-types. 
/**
   RP_NO: will be unloaded on request
   RP_LEVEL: will be unloaded at the end of a Level
   RP_CAMPAIGN: will be unloaded at the end of a Campaign
   RP_GAME: will be unloaded at the end of the whole Game (when closing orxonox)
*/
enum ResourcePriority {RP_NO = 0, RP_LEVEL = 1, RP_CAMPAIGN = 2, RP_GAME = 3};

//! A Struct that keeps track about A resource its name its Type, and so on
struct Resource
{
  void* pointer;             //!< Pointer to the Resource.
  
  char* name;                //!< Name of the Resource.
  ResourceType type;         //!< ResourceType of this Resource.
  ResourcePriority prio;     //!< The Priority of this resource. (This will only be increased)
  int count;                 //!< How many times this Resource has been loaded.
};


//! The ResourceManager is a class, that decides if a file/resource should be loaded
/**
   If a file/resource was already loaded the resourceManager will 
   return a void pointer to the desired resource.
   Otherwise it will instruct the corresponding resource-loader to load,
   and receive the pointer to it.

   It does it by looking, if a desired file has already been loaded.
*/
class ResourceManager : public BaseObject 
{
 public:
  static ResourceManager* getInstance();
  virtual ~ResourceManager();

  static bool setDataDir(char* dataDir);
  static bool addImageDir(char* imageDir);
  static void* load(const char* fileName, ResourcePriority prio = RP_NO);
  static void* load(const char* fileName, ResourceType type, ResourcePriority prio = RP_NO);
  static bool unload(void* pointer, ResourcePriority prio = RP_NO);
  static bool unload(Resource* resource, ResourcePriority = RP_NO);
  static bool unloadAllByPriority(ResourcePriority prio);

 private:
  ResourceManager();
  static ResourceManager* singletonRef;

  static tList<Resource>* resourceList;
  static char* dataDir;
  static tList<char>* imageDirs;

  static Resource* locateResourceByName(const char* fileName);
  static Resource* locateResourceByPointer(const void* pointer);
  
  static bool isDir(const char* directory);
  static bool isFile(const char* directory);  
};

#endif /* _RESOURCE_MANAGER_H */
