/* 
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_LOAD

#include "resource_manager.h"

// different resource Types
#include "objModel.h"
#include "texture.h"

// File Handling Includes
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>


using namespace std;


/**
   \brief standard constructor
*/
ResourceManager::ResourceManager () 
{
   this->setClassName ("ResourceManager");
   this->dataDir = NULL;
}

ResourceManager* ResourceManager::getInstance(void)
{
  if (!ResourceManager::singletonRef)
    ResourceManager::singletonRef = new ResourceManager();
  return ResourceManager::singletonRef;
}

ResourceManager* ResourceManager::singletonRef = NULL;

/**
   \brief standard deconstructor
*/
ResourceManager::~ResourceManager (void) 
{
  ResourceManager::singletonRef = NULL;
}



/**
   \brief sets the data main directory
   \param dataDir the DataDirectory.
*/
bool ResourceManager::setDataDir(char* dataDir)
{
  if (isDir(dataDir))
    {
      this->dataDir = new char[strlen(dataDir)+1];
      strcpy(this->dataDir, dataDir);
    }
  else
    {
      PRINTF(1)("%s is not a Directory, and can not be the Data Directory\n", dataDir);
    }
}

/**
   \brief loads resources
   \param fileName The fileName of the resource to load
   \returns a pointer to a desired Resource.
*/
void* ResourceManager::load(const char* fileName)
{
  ResourceType tmpType;
  if (!strncmp(fileName+(strlen(fileName)-4), ".obj", 4))
    tmpType = OBJ;

  return ResourceManager::load(fileName, tmpType);
}

/**
   \brief loads resources
   \param fileName The fileName of the resource to load
   \param type The Type of Resource to load (\see ResourceType)
   \returns a pointer to a desired Resource.
*/
void* ResourceManager::load(const char* fileName, ResourceType type)
{
  void* tmpResource = NULL;
  char* tmpName = new char[strlen(fileName)+1];
  strcpy(tmpName, fileName);
  // Checking for the type of resource \see ResourceType
  switch(type)
    {
    case OBJ:
      if(isFile(tmpName))
	tmpResource = new OBJModel(tmpName);
      else
	{
	  PRINTF(2)("Sorry, %s does not exist. Loading a cube-Model instead\n", tmpName);
	  tmpResource = new Model(CUBE);
	}
      break;
    case IMAGE:
      if(isFile(tmpName))
	{
	  tmpResource = new Texture(tmpName);
	}
      else
	PRINTF(1)("!!Texture %s not Found!!\n", tmpName);
      break;
    default:
      tmpResource = NULL;
      PRINTF(2)("No type found for %s.\n   !!This should not happen unless the Type is not supported yet.!!\n", tmpName);
      break;
    }

  // checking if the File really exists.
  if(!isFile(tmpName))
    {
      PRINTF(2)("Sorry, %s is not a regular file.\n", tmpName);
      tmpResource = NULL;
    }
  return tmpResource;
}

/**
   \brief Checks if it is a Directory
   \param directoryName the Directory to check for
   \returns true if it is a directory/symlink false otherwise
*/
bool ResourceManager::isDir(const char* directoryName)
{
  struct stat status;
  stat(directoryName, &status);
  if (status.st_mode & (S_IFDIR | S_IFLNK))
    return true;
  else 
    return false;
}

/**
   \brief Checks if the file is either a Regular file or a Symlink
   \param fileName the File to check for
   \returns true if it is a regular file/symlink, false otherwise
*/
bool ResourceManager::isFile(const char* fileName)
{
  struct stat status;
  stat(fileName, &status);
  if (status.st_mode & (S_IFREG | S_IFLNK))
    return true;
  else
    return false;
}
