/* 
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  


   ### File Specific:
   main-programmer: Patrick Boenzli
   co-programmer: Christian Meyer
*/

#include "orxonox.h"

#include "world.h"
#include "data_tank.h"
#include "command_node.h"
#include "game_loader.h"
#include "graphics_engine.h"

#include <string.h>
int verbose = 4;

using namespace std;

/**
   \brief create a new Orxonox
*/
Orxonox::Orxonox ()
{
  pause = false;
}

/**
   \brief remove Orxonox from memory
*/
Orxonox::~Orxonox () 
{
  Orxonox::singletonRef = NULL;
  if( world != NULL) delete world;
  if( localinput != NULL) delete world;
  if( resources != NULL) delete resources;
  delete GraphicsEngine::getInstance(); // deleting the Graphics
}

/** \brief this is a singleton class to prevent duplicates */
Orxonox* Orxonox::singletonRef = 0;

/**
   \returns reference or new Object of Orxonox if not existent.
*/
Orxonox* Orxonox::getInstance (void)
{
  if (singletonRef == NULL)
    singletonRef = new Orxonox();
  return singletonRef;
}

/**
   \brief this finds the config file
   
   Since the config file varies from user to user and since one may want to specify different config files
   for certain occasions or platforms this function finds the right config file for every occasion and stores
   it's path and name into configfilename
*/
void Orxonox::getConfigFile (int argc, char** argv)
{
  strcpy (configfilename, "orxonox.conf");
}

/**
   \brief initialize Orxonox with command line
*/
int Orxonox::init (int argc, char** argv)
{
  // parse command line
  // config file
  
  getConfigFile (argc, argv);
  SDL_Init (SDL_INIT_TIMER);
  // initialize everything
  if( initVideo() == -1) return -1;
  if( initSound() == -1) return -1;
  printf("> Initializing input\n");
  if( initInput() == -1) return -1;
  printf("> Initializing networking\n");
  if( initNetworking () == -1) return -1;
  printf("> Initializing resources\n");
  if( initResources () == -1) return -1;
  //printf("> Initializing world\n");
  //if( init_world () == -1) return -1; PB: world will be initialized when started
  
  return 0;
}

/**
   \brief initializes SDL and OpenGL
*/
int Orxonox::initVideo() 
{
  PRINTF(3)("> Initializing video\n");
  
  GraphicsEngine::getInstance();
  
  return 0;
}


/**
   \brief initializes the sound engine
*/
int Orxonox::initSound() 
{
  printf("> Initializing sound\n");
  // SDL_Init(SDL_INIT_AUDIO);
  printf("Not yet implemented\n");
  return 0;
}


/**
   \brief initializes input functions
*/
int Orxonox::initInput() 
{
  // create localinput
  localinput = new CommandNode( configfilename);
  
  return 0;
}


/**
   \brief initializes network system
*/
int Orxonox::initNetworking() 
{
  printf("Not yet implemented\n");
  return 0;
}


/**
   \brief initializes and loads resource files
*/
int Orxonox::initResources() 
{
  printf("Not yet implemented\n");
  return 0;
}


/**
   \brief initializes the world
*/
int Orxonox::initWorld() 
{
  //world = new World();
  
  // TO DO: replace this with a menu/intro
  //world->load_debug_level();
  
  return 0;
}


/**
   \brief starts the orxonox game or menu

   here is the central orxonox state manager. There are currently two states
   - menu
   - game-play
   both states manage their states themselfs again.
*/
void Orxonox::start()
{
  
  this->gameLoader = GameLoader::getInstance();
  this->gameLoader->loadDebugCampaign(DEBUG_CAMPAIGN_0);
  this->gameLoader->init();
  this->gameLoader->start();
}


/**
   \brief exits Orxonox
*/
void Orxonox::quitGame() 
{
  bQuitOrxonox = true;
}



/**
   \brief handles sprecial events from localinput
   \param event: an event not handled by the CommandNode 
*/
void Orxonox::eventHandler(SDL_Event* event)
{
  // Handle special events such as reshape, quit, focus changes
  switch (event->type)
    {
    case SDL_VIDEORESIZE:
      GraphicsEngine* tmpGEngine = GraphicsEngine::getInstance();
      tmpGEngine->resolutionChanged(&event->resize);
      break;
    }
}
 

/**
   \brief handle keyboard commands that are not meant for WorldEntities
   \param cmd: the command to handle
   \return true if the command was handled by the system or false if it may be passed to the WorldEntities
*/
bool Orxonox::systemCommand(Command* cmd)
{
  /*
  if( !strcmp( cmd->cmd, "quit"))
    {
      if( !cmd->bUp) this->gameLoader->stop();
      return true;
    }
  return false;
  */
  return false;
}

/**
   \brief retrieve a pointer to the local CommandNode
   \return a pointer to localinput
*/
CommandNode* Orxonox::getLocalInput()
{
  return localinput;
}


/**
   \brief retrieve a pointer to the local World
   \return a pointer to world
*/
World* Orxonox::getWorld()
{
  return world;
}

/**
   \return The reference of the SDL-screen of orxonox
*/
SDL_Surface* Orxonox::getScreen ()
{
  return this->screen;
}


#include "list.h"
#include "world_entity.h"
#include "vector.h"
#include <asm/msr.h>
#include <linux/timex.h>


#define LIST_MAX 10000
#define VECTOR_MAX 1000000
#define ITERATIONS 10000

/**
   \brief main function

   here the journey begins
*/
int main(int argc, char** argv) 
{  
  
  /* reading arguments 
     
     currently supported arguments are:
     <no args>                   ::    just starts orxonox
     --benchmark                 ::    start the benchmark without starting orxonox
     
     this is a preselection: it matches to one of the start* functions, the
     finetuning is made in those functions.
  */


  int i;
  for(i = 0; i < argc; ++i)
    {
      if(! strcmp( "--help", argv[i])) return startHelp();
      else if(! strcmp( "--benchmark", argv[i])) return startBenchmarks();
    }

  PRINTF(2)("Orxonox does not understand the arguments");
  return startOrxonox(argc, argv);
}



int startHelp()
{
  printf("orxonox: starts the orxonox game - rules\n");
  printf("usage: orxonox [arg]\n\n");
  printf("valid options:\n");
  printf(" --benchmark\tstarts the orxonox benchmark\n");
  printf(" --help \tshows this menu\n");
}

int startOrxonox(int argc, char** argv)
{
  printf(">>> Starting Orxonox <<<\n");
  Orxonox *orx = Orxonox::getInstance();
  
  if((*orx).init(argc, argv) == -1)
    {
      printf("! Orxonox initialization failed\n");
      return -1;
    }
  
  orx->start();
  
  //delete orx;
  
}


int startBenchmarks()
{

  printf("===========================================================\n");
  printf("=                      BENCHMARKS                         =\n");
  printf("===========================================================\n");

  // first measure the time overhead:
  unsigned long ini, end, dt, tmp;
  rdtscl(ini); rdtscl(end);
  dt = end - ini;

  int type = 0; 
  /* type   -1 == all 
     type    0 == framework
     type    1 == vector
     type    2 == quaternion
  */
  if(type == 0 || type == -1)
    {
      /* framework test*/
      WorldEntity* w = NULL;
      int i = 0;
      unsigned long mittel = 0;
      
      for(i = 0; i < ITERATIONS; ++i)
	{
	  rdtscl(ini);
	  
	  WorldEntity* w = new WorldEntity();
	  
	  rdtscl(end);
	  mittel += (end - ini - dt);
	}
      float mi = mittel / (float)ITERATIONS;
      printf("Generate a WorldEntity objec [cycles]t: \t%f\n", mi);
      
      
      mittel = 0;
      w = new WorldEntity();
      for(i = 0; i < ITERATIONS; ++i)
	{
	  rdtscl(ini);
	  
	  w->tick(0.0f);
	   
	  rdtscl(end);
	  mittel += (end - ini - dt);
	  }
      mi = mittel / (float)ITERATIONS;
      printf("Call an empty function of WE [cycles]t: \t%f\n", mi);
      
      tList<WorldEntity>* list = new tList<WorldEntity>();
      
      
      // generating list
      for(i = 0; i < LIST_MAX; ++i)
	{
	  w = new WorldEntity();
	  list->add(w);
	}
	
      
	// reuse list
      w = list->enumerate();
      while( w != NULL)
	{
	  w->tick(0.0f);
	  w = list->nextElement();
	}

      }
  if(type == 1 || type == -1)
    {
      /* vector test */
      Vector* a = new Vector(1.3, 5.3, 4.1);
      Vector* b = new Vector(0.4, 2.5, 6.2);
      Vector* c = new Vector();
      
      int i = 0;
      // addition
      for(i = 0; i < VECTOR_MAX; ++i)
	{
	  *c = *a + *b;
	}
      
      // multiplikation
      for(i = 0; i < VECTOR_MAX; ++i)
	{
	  *c = a->cross( *b);
	}
      if(type == 1 || type == -1)
	{
	  /* quaternion test */

	}
      

    }
}
