/*!
    \file command_node.h
    \brief Parses keyboard, mouse and remote input
    
    Contains methods to parse remote and local input and handles sending of input to remote CommandNodes
*/

#ifndef _COMMAND_NODE_H
#define _COMMAND_NODE_H


#include "comincl.h"

template<class T> class tList;
class WorldEntity;
class World;

#define N_STD_KEYS SDLK_LAST
#define N_BUTTONS 6
#define DEFAULT_KEYBIND_FILE "default.ini"

//! Key aliasing structure
/**
   This structure contains the key aliasing information, e.g. the command strings that
   have been bound to the keys.
*/
typedef struct
{
  char keys[N_STD_KEYS][CMD_LENGHT];
  char buttons[N_BUTTONS][CMD_LENGHT];
} KeyBindings;

//! Command Node
/**
   This class gathers all incoming SDL_Events and processes them. Keyboard, mouse and joystick input is
   captured and translated into command messages which are passed down to the bound WorldEntities (via WorldEntity::command()).
   Other SDL_Events are passed to Orxonox::event_handler() to deal with them. If the CommandNode has been created
   with bLocalInput set to false, it will query the network class for incoming commands that match his netID and pass
   them on to it's WorldEntities.
*/
class CommandNode {
 private:
  bool bLocalInput;	//!< Identifies the CommandNode that processes local input
  bool bEnabled;
  int netID;	//!< Unique identifier that is used to determine between remote CommandNodes
  KeyBindings* aliases;
  tList<WorldEntity>* bound;	//!< List of WorldEntites that recieve commands from this CommandNode
  Sint32 coord[2];
  World* world;
  

  void relay (Command* cmd);
  int* nameToIndex (char* name);
  void processLocal ();
  void processNetwork ();
  void sendOverNetwork (Command* cmd);
  
 public:
  CommandNode (int ID);
  CommandNode (char* filename);
  ~CommandNode ();

  void reset ();
  void enable (bool bEnabled);
  void loadBindings (char* filename);
  void bind (WorldEntity* entity);
  void unbind (WorldEntity* entity);
  void addToWorld (World* world);
  void process ();
  
  void setNetID (int ID);
};

#endif /* _COMMAND_NODE_H */
