/*! 
    \file vector.h
    \brief A basic 3D math framework
    
    Contains classes to handle vectors, lines, rotations and planes
*/ 

#ifndef _VECTOR_H
#define _VECTOR_H

#include <math.h>
//! PI the circle-constant
#define PI 3.14159265359f  /* this is not nice... make a math_def file or something */ 

//! 3D Vector
/**
	Class to handle 3D Vectors
*/
class Vector {

  public:
 
  float x; //!< The x Coordinate of the Vector.
  float y; //!< The y Coordinate of the Vector.
  float z; //!< The z Coordinate of the Vector.

  Vector (float x, float y, float z) : x(x), y(y), z(z) {}  //!< assignment constructor
  Vector () : x(0), y(0), z(0) {}
  ~Vector () {}

  Vector operator+ (const Vector& v) const;
  Vector operator- (const Vector& v) const;
  float operator* (const Vector& v) const;
  Vector operator* (float f) const;
  Vector operator/ (float f) const;
  inline float dot (const Vector& v) const;
  Vector cross (const Vector& v) const;
  void scale(const Vector& v);
  float len() const;
  void normalize();
  Vector* getNormalized();
  Vector abs();

  void debug();
};

//float angleDeg (const Vector& v1, const Vector& v2);
//float angleRad (const Vector& v1, const Vector& v2);


/**
   \brief add two vectors
   \param v: the other vector
   \return the sum of both vectors
*/
Vector Vector::operator+ (const Vector& v) const
{
  Vector r;

  r.x = x + v.x;
  r.y = y + v.y;
  r.z = z + v.z;
  
  return r;
}

/**
   \brief subtract a vector from another
   \param v: the other vector
   \return the difference between the vectors
*/
Vector Vector::operator- (const Vector& v) const
{
  Vector r;
  
  r.x = x - v.x;
  r.y = y - v.y;
  r.z = z - v.z;
  
  return r;
}

/**
   \brief calculate the dot product of two vectors
   \param v: the other vector
   \return the dot product of the vectors
*/
float Vector::operator* (const Vector& v) const
{
  return x * v.x + y * v.y + z * v.z;
}

/**
   \brief multiply a vector with a float
   \param f: the factor
   \return the vector multipied by f
*/
Vector Vector::operator* (float f) const
{  
  Vector r;
  
  r.x = x * f;
  r.y = y * f;
  r.z = z * f;
  
  return r;
}

/**
   \brief divide a vector with a float
   \param f: the divisor
   \return the vector divided by f   
*/
Vector Vector::operator/ (float f) const
{
  Vector r;
  
  if( f == 0.0)
  {
    // Prevent divide by zero
    return Vector (0,0,0);
  }
  
  r.x = x / f;
  r.y = y / f;
  r.z = z / f;
  
  return r;
}

/**
   \brief calculate the dot product of two vectors
   \param v: the other vector
   \return the dot product of the vectors
*/
inline float Vector::dot (const Vector& v) const
{
  return x*v.x+y*v.y+z*v.z;
}

/**
  \brief calculate the cross product of two vectors
  \param v: the other vector
	\return the cross product of the vectors
*/
Vector Vector::cross (const Vector& v) const
{
  Vector r;
  
  r.x = y * v.z - z * v.y;
  r.y = z * v.x - x * v.z;
  r.z = x * v.y - y * v.x;
  
  return r;
}
  
/**
   \brief normalizes the vector to lenght 1.0
*/
void Vector::normalize ()
{
  float l = len();
  
  if( l == 0.0)
  {
    // Prevent divide by zero
    return;
  }
  
  x = x / l;
  y = y / l;
  z = z / l;
}


/**
   \brief returns the voctor normalized to length 1.0
*/

Vector* Vector::getNormalized()
{
  float l = len();
  if(l != 1.0)
    {
      return this;
    }
  else if(l == 0.0)
    {
      return 0;
    }
  
  Vector *normalizedVector = new Vector(x / l, y /l, z / l);
  return normalizedVector;
}

/**
   \brief scales this Vector with Vector v.
   \param v the vector to scale this vector with
*/
void Vector::scale(const Vector& v)
{
  x *= v.x;
  y *= v.y;
  z *= v.z;
}

 
/**
   \brief calculates the lenght of the vector
   \return the lenght of the vector
*/
float Vector::len () const
{
  return sqrt (x*x+y*y+z*z);
}


/**
   \brief Vector is looking in the positive direction on all axes after this
*/
Vector Vector::abs() 
{
  Vector v(fabs(x), fabs(y), fabs(z));
  return v;
}

/**
   \brief calculate the angle between two vectors in radiances
   \param v1: a vector
   \param v2: another vector
   \return the angle between the vectors in radians
*/
float angleRad (const Vector& v1, const Vector& v2)
{
  return acos( v1 * v2 / (v1.len() * v2.len()));
}


/**
   \brief calculate the angle between two vectors in degrees
   \param v1: a vector
   \param v2: another vector
   \return the angle between the vectors in degrees
*/
float angleDeg (const Vector& v1, const Vector& v2)
{
  float f;
  f = acos( v1 * v2 / (v1.len() * v2.len()));
  return f * 180 / PI;
}


/**
   \brief Outputs the values of the Vector
*/
void Vector::debug(void)
{
  PRINT(0)("Vector Debug information\n");
  PRINT(0)("x: %f; y: %f; z: %f", x, y, z);
  PRINT(3)(" lenght: %f", len());
  PRINT(0)("\n");
}


#endif /* _VECTOR_H */
