/*! 
    \file light.h
    \brief Handles Lights.

    A Light is one of the more important things in a 3D-environment,
    without it one sees nothing :)
    It is here for diffuse-, specular- and Bump-Mappings.
*/

#ifndef _LIGHT_H
#define _LIGHT_H

#include "world_entity.h"
#include "glincl.h"

//! The maximum number of Lights this OpenGL-implementation supports.
#define NUMBEROFLIGHTS GL_MAX_LIGHTS

// FORWARD DEFINITIONS //
class Vector;

//! A class that handles Lights. The LightManager operates on this.
class Light : public WorldEntity
{
 public:
  Light(int lightNumber);
  virtual ~Light(void);

  void setPosition(Vector position);
  void setPosition(GLfloat x, GLfloat y, GLfloat z);
  void setDiffuseColor(GLfloat r, GLfloat g, GLfloat b);
  void setSpecularColor(GLfloat r, GLfloat g, GLfloat b);
  void setAttenuation(float constantAttenuation, float linearAttenuation, float quadraticAttenuation);
  void setSpotDirection(Vector direction);
  void setSpotCutoff(GLfloat cutoff);

  Vector getPosition() const;
  /** \returns the lightNumber*/
  int getLightNumber(void) const {return this->lightNumber;}
  
  virtual void draw();

  void debug(void) const;

  // attributes
 private:
  int lightNumber;            //!< The number of this Light.
  GLfloat lightPosition[4];   //!< The Position of this Light.
  GLfloat diffuseColor[4];    //!< The Diffuse Color this Light emmits.
  GLfloat specularColor[4];   //!< The specular Color of this Light.
  float constantAttenuation;  //!< The Factor of the the Constant Attenuation.
  float linearAttenuation;    //!< The Factor of the the Linear Attenuation.
  float quadraticAttenuation; //!< The Factor of the the Quadratic Attenuation.
  GLfloat spotDirection[4];   //!< The direction of the Spot Light.
  GLfloat spotCutoff;         //!< The cutoff Angle of the Light Source
};

//! A class that handles Lights
/**
   A Light is a source that emits light rays (photons)

   <b>Usage:</b>\n
   First you have to get the Light Manager up and running by using LightManager::getInstance.
   This automatically initiates the GL_LIGHTING, and sets some default stuff about the light.\n
   Then you will create a new light using:
   \li int addLight(void);
   \li int addLight(int lightNumber);

   now you can operate on the light as follows:
   \li void setPosition(Vector position);
   \li void setPosition(GLfloat x, GLfloat y, GLfloat z);
   \li void setDiffuseColor(GLfloat r, GLfloat g, GLfloat b);
   \li void setSpecularColor(GLfloat r, GLfloat g, GLfloat b);
   \li void setAttenuation(float constantAttenuation, float linearAttenuation, float quadraticAttenuation);
   \li void setSpotDirection(Vector direction);
   \li void setSpotCutoff(GLfloat cutoff);

   These functions are preatty selv-explaining, but you can read about them below.\n
   If you want to fetch info, or the a Light use:
   \li Vector getPosition(void) const;
   \li Vector getPosition(int lightNumber) const;
   \li Light* getLight(int LightNumber) const;

   To redraw the light use
   \li void draw();
   
   and to delete one just use
   \li void deleteLight(void);

   for some nice output just use:
   \li void debug(void) const;

   You can also operate on the single Lights themselves, by first retreaving them with 
   \li Light* getLight(int LightNumber);
   \n
   and then using the same functions on this to change settings.
*/
class LightManager : public BaseObject
{
 private:
  LightManager(void);
  void initLight(int LightNumber);
  
  static LightManager* singletonRef;    //!< This is the LightHandlers Reference.
  GLfloat ambientColor[4];       //!< The ambient Color of the scene.

  Light** lights;                //!< An array of Lenght NUMBEROFLIGHTS, that holds pointers to all LightValues.
  Light* currentLight;           //!< The current Light, we are working with.
  
 public:
  static LightManager* getInstance();
  virtual ~LightManager(void);

  // set Attributes
  int addLight(void);
  int addLight(int lightNumber);
  void editLightNumber(int lightNumber);
  void deleteLight(void);
  void deleteLight(int lightNumber);

  void setAmbientColor(GLfloat r, GLfloat g, GLfloat b);

  void draw();

  //set Attributes
  void setPosition(Vector position);
  void setPosition(GLfloat x, GLfloat y, GLfloat z);
  void setDiffuseColor(GLfloat r, GLfloat g, GLfloat b);
  void setSpecularColor(GLfloat r, GLfloat g, GLfloat b);
  void setAttenuation(float constantAttenuation, float linearAttenuation, float quadraticAttenuation);
  void setSpotDirection(Vector direction);
  void setSpotCutoff(GLfloat cutoff);

  // get Attributes
  Vector getPosition(void) const;
  Vector getPosition(int lightNumber) const;
  
  Light* getLight(int lightNumber) const;
  
  void debug(void) const;
};

#endif /* _LIGHT_H */
