
/* 
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Patrick Boenzli
   co-programmer: Christian Meyer
*/

#include "world.h"
#include "world_entity.h"
#include "track_manager.h"
#include "player.h"
#include "command_node.h"
#include "camera.h"
#include "environment.h"
#include "primitive.h"
#include "p_node.h"
#include "null_parent.h"
#include "helper_parent.h"
#include "glmenu_imagescreen.h"
#include "skysphere.h"
#include "light.h"
#include "fontset.h"
#include "track_node.h"
#include "terrain.h"

using namespace std;


/** 
    \brief create a new World
    
    This creates a new empty world!
*/
World::World (char* name)
{
  this->init(name, -1);
  //NullParent* np = NullParent::getInstance();
}

/**
   \brief creates a new World...
   \param worldID with this ID
*/
World::World (int worldID)
{
  this->init(NULL, worldID);
}

/** 
    \brief remove the World from memory
    
    delete everything explicitly, that isn't contained in the parenting tree!
    things contained in the tree are deleted automaticaly
*/
World::~World ()
{
  PRINTF(3)("World::~World() - deleting current world\n");
  CommandNode* cn = Orxonox::getInstance()->getLocalInput();
  cn->unbind(this->localPlayer);
  cn->reset();

  delete this->nullParent;
  delete this->entities;
  delete this->light;
  delete this->trackManager;
}

/**
   \brief initializes a new World
*/
void World::init(char* name, int worldID)
{
  this->setClassName ("World");

  this->worldName = name;
  this->debugWorldNr = worldID;
  this->entities = new tList<WorldEntity>();

  // Enable default GL stuff
  glEnable(GL_DEPTH_TEST);

}


/**
   \brief loads the World by initializing all resources, and set their default values.
*/
ErrorMessage World::load()
{
  //  BezierCurve* tmpCurve = new BezierCurve();
  if(this->debugWorldNr != -1)
    {
      // initializing Font
      testFont = new FontSet();
      testFont->buildFont("../data/pictures/font.tga");

      // initializing the TrackManager
      trackManager = TrackManager::getInstance();
      trackManager->addPoint(Vector(0,0,0));
      trackManager->addPoint(Vector(100, -40, 5));
      trackManager->addPoint(Vector(200,-40,-8));
      trackManager->addPoint(Vector(250, -35, -2));
      trackManager->addPoint(Vector(320,-33,-.55));
      trackManager->setDuration(1);
      trackManager->setSavePoint();
      trackManager->addPoint(Vector(410, 0, 0));
      trackManager->addPoint(Vector(510, 20, -10));
      trackManager->addPoint(Vector(550, 20, -10));
      trackManager->addPoint(Vector(570, 20, -10));
      trackManager->setDuration(1);
      
      int fork11, fork12;
      trackManager->fork(2, &fork11, &fork12);
      trackManager->workOn(fork11);
      trackManager->addPoint(Vector(640, 25, -30));
      trackManager->addPoint(Vector(700, 40, -120));
      trackManager->addPoint(Vector(800, 50, -150));
      trackManager->addPoint(Vector(900, 60, -100));
      trackManager->addPoint(Vector(900, 60, -70));
      trackManager->addPoint(Vector(990, 65, -15));
      trackManager->addPoint(Vector(1050, 65, -10));
      trackManager->addPoint(Vector(1100, 65, -20));
      trackManager->setDuration(4);

      trackManager->workOn(fork12);
      trackManager->addPoint(Vector(640, 25, 20));
      trackManager->addPoint(Vector(670, 50, 120));
      trackManager->addPoint(Vector(700, 70, 80));
      trackManager->addPoint(Vector(800, 70, 65));
      trackManager->addPoint(Vector(850, 65, 65));
      trackManager->addPoint(Vector(920, 35, 40));
      trackManager->addPoint(Vector(945, 40, 40));
      trackManager->addPoint(Vector(970, 24, 40));
      trackManager->addPoint(Vector(1000, 40, -7));
      trackManager->setDuration(4);
      

      trackManager->join(2, fork11, fork12);

      trackManager->workOn(5);
      trackManager->addPoint(Vector(1200, 60, -50));
      trackManager->addPoint(Vector(1300, 50, -50));
      trackManager->addPoint(Vector(1400, 40, -50));
      trackManager->addPoint(Vector(1500, 40, -60));
      trackManager->addPoint(Vector(1600, 35, -55));
      trackManager->addPoint(Vector(1700, 45, -40));
      trackManager->addPoint(Vector(1750, 60, -40));
      trackManager->addPoint(Vector(1770, 80, -40));
      trackManager->addPoint(Vector(1800, 100, -40));
      trackManager->setDuration(4);

      trackManager->finalize();

      
      /*monitor progress*/
      this->glmis->step();

      switch(this->debugWorldNr)
	{
	  /*
	    this loads the hard-coded debug world. this only for simplicity and will be 
	    removed by a reald world-loader, which interprets a world-file.
	    if you want to add an own debug world, just add a case DEBUG_WORLD_[nr] and
	    make whatever you want...
	   */
	case DEBUG_WORLD_0:
	  {
	    this->nullParent = NullParent::getInstance ();
	    this->nullParent->setName ("NullParent");

	    // !\todo old track-system has to be removed

	    //create helper for player
	    //HelperParent* hp = new HelperParent ();
	    /* the player has to be added to this helper */

	    // create a player
	    this->localPlayer = new Player ();
	    this->localPlayer->setName ("player");
	    this->spawn (this->localPlayer);
	    /*monitor progress*/
	    this->glmis->step();	    

	    // bind input
	    Orxonox *orx = Orxonox::getInstance ();
	    orx->getLocalInput()->bind (this->localPlayer);
	    
	    // bind camera
	    this->localCamera = new Camera(this);
	    this->localCamera->setName ("camera");
	    this->localCamera->bind (this->localPlayer);
	    /*monitor progress*/
	    this->glmis->step();	    

	    // Create SkySphere
	    this->skySphere = new Skysphere("../data/pictures/sky-replace.jpg");
	    this->skySphere->setName("SkySphere");
	    this->localCamera->addChild(this->skySphere);
	    this->skySphere->setMode(PNODE_MOVEMENT);

	    /*monitor progress*/
	    this->glmis->step();

	    
	    WorldEntity* env = new Environment();
	    env->setName ("env");
	    this->spawn(env);

	   
	    Vector* es = new Vector (10, 5, 0);
	    Quaternion* qs = new Quaternion ();
	    WorldEntity* pr = new Primitive(PSPHERE);
	    pr->setName("primitive");
	    this->spawn(pr, this->localPlayer, es, qs, PNODE_MOVEMENT);
	   

	    /*monitor progress*/
	    this->glmis->step();

	    //	    trackManager->setBindSlave(env);
	    PNode* tn = trackManager->getTrackNode();
	    tn->addChild(this->localPlayer);

	    //localCamera->setParent(TrackNode::getInstance());
	    tn->addChild (this->localCamera);
	    this->localPlayer->setMode(PNODE_ROTATE_AND_MOVE);
	    //Vector* cameraOffset = new Vector (0, 5, -10);
	    Vector* cameraOffset = new Vector (-10, 5, 0);
	    this->localCamera->setRelCoor (cameraOffset);
	    trackManager->condition(2, NEAREST, this->localPlayer);

	    break;
	  }
	case DEBUG_WORLD_1:
	  {
	    this->nullParent = NullParent::getInstance ();
	    this->nullParent->setName ("NullParent");

	    // create a player
	    WorldEntity* myPlayer = new Player();
	    myPlayer->setName ("player");
	    this->spawn(myPlayer);
	    this->localPlayer = myPlayer;	    
	    
	    // bind input
	    Orxonox *orx = Orxonox::getInstance();
	    orx->getLocalInput()->bind (myPlayer);
	    
	    // bind camera
	    this->localCamera = new Camera (this);
	    this->localCamera->setName ("camera");
	    this->localCamera->bind (myPlayer);	
	    this->localPlayer->addChild (this->localCamera);

	    // Create SkySphere
	    skySphere = new Skysphere("../data/pictures/sky-replace.jpg");
	    this->localPlayer->addChild(this->skySphere);

	    Vector* es = new Vector (20, 0, 0);
	    Quaternion* qs = new Quaternion ();
	    WorldEntity* pr = new Primitive(PSPHERE);
	    pr->setName("primitive");
	    this->spawn(pr, this->localPlayer, es, qs, PNODE_ROTATE_AND_MOVE);

	    break;
	  }
	default:
	  printf("World::load() - no world with ID %i found", this->debugWorldNr );
	}
    }
  else if(this->worldName != NULL)
    {

    }

  // initialize debug coord system
  objectList = glGenLists(1);
  glNewList (objectList, GL_COMPILE);
 
  trackManager->drawGraph(.01);
  trackManager->debug(2);
  glEndList();

  terrain = new Terrain("../data/worlds/newGround.obj");
  terrain->setRelCoor(new Vector(0,-10,0));
  this->spawn(terrain);
  // LIGHT initialisation
  light = Light::getInstance();
  light->setAmbientColor(.1,.1,.1);
  light->addLight();
  light->setAttenuation(QUADRATIC, 1.0);
  light->setAttenuation(CONSTANT, 2.0);
  light->setAttenuation(QUADRATIC, 1.0);
  light->setPosition(10.0, 30.0, 10.0);
  light->setDiffuseColor(1,1,1);
  //  light->addLight(1);
  //  light->setPosition(20, 10, -20);
  //  light->setDiffuseColor(0,0,0);
  light->debug();


}

/**
   \brief initializes a new World
*/
ErrorMessage World::init()
{
  this->bPause = false;
  CommandNode* cn = Orxonox::getInstance()->getLocalInput();
  cn->addToWorld(this);
  cn->enable(true);

}


/**
   \brief starts the World
*/
ErrorMessage World::start()
{
  PRINTF(3)("World::start() - starting current World: nr %i\n", this->debugWorldNr);
  this->bQuitOrxonox = false;
  this->bQuitCurrentGame = false;
  this->mainLoop();
}

/**
   \brief stops the world.

   This happens, when the player decides to end the Level.
*/
ErrorMessage World::stop()
{
  PRINTF(3)("World::stop() - got stop signal\n");
  this->bQuitCurrentGame = true;
}

/**
   \brief pauses the Game
*/
ErrorMessage World::pause()
{
  this->isPaused = true;
}

/**
   \brief ends the pause Phase
*/
ErrorMessage World::resume()
{
  this->isPaused = false;
}

/**
   \brief destroys the World
*/
ErrorMessage World::destroy()
{

}

/**
   \brief shows the loading screen
*/
void World::displayLoadScreen ()
{
  PRINTF(3)("World::displayLoadScreen - start\n"); 
  
  //GLMenuImageScreen* 
  this->glmis = GLMenuImageScreen::getInstance();
  this->glmis->init();
  this->glmis->setMaximum(10);
  this->glmis->draw();
 
  PRINTF(3)("World::displayLoadScreen - end\n"); 
}

/**
   \brief removes the loadscreen, and changes over to the game

   \todo take out the delay
*/
void World::releaseLoadScreen ()
{
  PRINTF(3)("World::releaseLoadScreen - start\n"); 
  this->glmis->setValue(this->glmis->getMaximum());
  SDL_Delay(500);
  PRINTF(3)("World::releaseLoadScreen - end\n"); 
}


/** 
    \brief checks for collisions
    
    This method runs through all WorldEntities known to the world and checks for collisions 
    between them. In case of collisions the collide() method of the corresponding entities 
    is called.
*/
void World::collide ()
{
  /*
  List *a, *b;
  WorldEntity *aobj, *bobj;
   
  a = entities;
  
  while( a != NULL)
    {
      aobj = a->nextElement();
      if( aobj->bCollide && aobj->collisioncluster != NULL)
	{
	  b = a->nextElement();
	  while( b != NULL )
	    {
	      bobj = b->nextElement();
	      if( bobj->bCollide && bobj->collisioncluster != NULL )
		{
		  unsigned long ahitflg, bhitflg;
		  if( check_collision ( &aobj->place, aobj->collisioncluster, 
					&ahitflg, &bobj->place, bobj->collisioncluster, 
					&bhitflg) );
		  {
		    aobj->collide (bobj, ahitflg, bhitflg);
		    bobj->collide (aobj, bhitflg, ahitflg);
		  }
		}
	      b = b->nextElement();
	    }
	}
      a = a->enumerate();
    }
  */
}

/** 
    \brief runs through all entities calling their draw() methods
*/
void World::draw ()
{
  /* draw entities */
  WorldEntity* entity;
  glLoadIdentity();

  entity = this->entities->enumerate();
  while( entity != NULL ) 
    { 
      if( entity->bDraw ) entity->draw();
      entity = this->entities->nextElement();
    } 
  
  glCallList (objectList);
  //! \todo skysphere is a WorldEntity and should be inside of the world-entity-list.
  skySphere->draw();

  testFont->printText(0, 0, 1, "orxonox_" PACKAGE_VERSION);

}


/**
   \brief function to put your own debug stuff into it. it can display informations about
   the current class/procedure
*/
void World::debug()
{
  PRINTF(2)("debug() - starting debug\n");
  PNode* p1 = NullParent::getInstance ();
  PNode* p2 = new PNode (new Vector(2, 2, 2), p1);
  PNode* p3 = new PNode (new Vector(4, 4, 4), p1);
  PNode* p4 = new PNode (new Vector(6, 6, 6), p2);

  p1->debug ();
  p2->debug ();
  p3->debug ();
  p4->debug ();

  p1->shiftCoor (new Vector(-1, -1, -1));

  printf("World::debug() - shift\n");
  p1->debug ();
  p2->debug ();
  p3->debug ();
  p4->debug ();
  
  p1->update ();

  printf ("World::debug() - update\n");
  p1->debug ();
  p2->debug ();
  p3->debug ();
  p4->debug ();

  p2->shiftCoor (new Vector(-1, -1, -1));
  p1->update ();

  p1->debug ();
  p2->debug ();
  p3->debug ();
  p4->debug ();

  p2->setAbsCoor (new Vector(1,2,3));


 p1->update ();

  p1->debug ();
  p2->debug ();
  p3->debug ();
  p4->debug ();

  delete p1;
  
  
  /*
  WorldEntity* entity;
  printf("counting all entities\n");
  printf("World::debug() - enumerate()\n");
  entity = entities->enumerate();  
  while( entity != NULL ) 
    { 
      if( entity->bDraw ) printf("got an entity\n");
      entity = entities->nextElement();
    }
  */
}


/**
  \brief main loop of the world: executing all world relevant function

  in this loop we synchronize (if networked), handle input events, give the heart-beat to
  all other member-entities of the world (tick to player, enemies etc.), checking for
  collisions drawing everything to the screen.
*/
void World::mainLoop()
{
  this->lastFrame = SDL_GetTicks ();
  PRINTF(3)("World::mainLoop() - Entering main loop\n");
  while( !this->bQuitOrxonox && !this->bQuitCurrentGame) /* \todo implement pause */
    {
      PRINTF(3)("World::mainloop() - number of entities: %i\n", this->entities->getSize());
      // Network
      this->synchronize ();
      // Process input
      this->handleInput ();
      if( this->bQuitCurrentGame || this->bQuitOrxonox)
	  break;
      // Process time
      this->tick ();
      // Update the state
      this->update ();      
      // Process collision
      this->collide ();
      // Draw
      this->display ();

      //      for( int i = 0; i < 5000000; i++) {}
      /* \todo this is to slow down the program for openGl Software emulator computers, reimplement*/
    }
  PRINTF(3)("World::mainLoop() - Exiting the main loop\n");
}


/**
   \brief synchronize local data with remote data
*/
void World::synchronize ()
{
  // Get remote input
  // Update synchronizables
}


/**
   \brief run all input processing

   the command node is the central input event dispatcher. the node uses the even-queue from
   sdl and has its own event-passing-queue.
*/
void World::handleInput ()
{
  // localinput
  CommandNode* cn = Orxonox::getInstance()->getLocalInput();
  cn->process();
  // remoteinput
}


/**
   \brief advance the timeline

   this calculates the time used to process one frame (with all input handling, drawing, etc)
   the time is mesured in ms and passed to all world-entities and other classes that need
   a heart-beat.
*/
void World::tick ()
{
  Uint32 currentFrame = SDL_GetTicks();
  if(!this->bPause)
    {
      Uint32 dt = currentFrame - this->lastFrame;
      
      if(dt > 0)
	{
	  float fps = 1000/dt;
	  PRINTF(3)("fps = %f\n", fps);
	}
      else
	{
	  /* the frame-rate is limited to 100 frames per second, all other things are for
	     nothing.
	  */
	  PRINTF(2)("fps = 1000 - frame rate is adjusted\n");
	  SDL_Delay(10);
	  dt = 10;
	}
      //this->timeSlice (dt);
      
      /* function to let all entities tick (iterate through list) */
      WorldEntity* entity;
      float seconds = dt / 1000.0;      
      entity = entities->enumerate(); 
      while( entity != NULL) 
	{ 
	  entity->tick (seconds);
	  entity = entities->nextElement();
	}
      //skySphere->updatePosition(localCamera->absCoordinate);
      
      /* update tick the rest */
      this->localCamera->tick(dt);
      this->trackManager->tick(dt);
    }
  this->lastFrame = currentFrame;
}


/**
   \brief this function gives the world a consistant state

   after ticking (updating the world state) this will give a constistant
   state to the whole system.
*/
void World::update()
{
  this->nullParent->update ();
}


/**
   \brief render the current frame
   
   clear all buffers and draw the world
*/
void World::display ()
{
  // clear buffer
  glClear( GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);
  // set camera
  this->localCamera->apply ();
  // draw world
  this->draw();
  // draw HUD
  /* \todo draw HUD */
  // flip buffers
  SDL_GL_SwapBuffers();
  //SDL_Surface* screen = Orxonox::getInstance()->getScreen ();
  //SDL_Flip (screen);
}


/**
   \brief add and spawn a new entity to this world
   \param entity to be added
*/
void World::spawn(WorldEntity* entity)
{
  this->entities->add (entity);
  entity->postSpawn ();
}


/**
   \brief add and spawn a new entity to this world
   \param entity to be added
   \param absCoor At what coordinates to add this entity.
   \param absDir In which direction should it look.
*/
void World::spawn(WorldEntity* entity, Vector* absCoor, Quaternion* absDir)
{
  this->entities->add (entity);

  entity->setAbsCoor (absCoor);
  entity->setAbsDir (absDir);

  entity->postSpawn ();
}


/**
   \brief add and spawn a new entity to this world
   \param entity to be added
   \param entity to be added to (PNode)
   \param At what relative  coordinates to add this entity.
   \param In which relative direction should it look.
*/
void World::spawn(WorldEntity* entity, PNode* parentNode, 
		  Vector* relCoor, Quaternion* relDir, 
		  int parentingMode)
{
  this->nullParent = NullParent::getInstance();
  if( parentNode != NULL)
    {
      parentNode->addChild (entity);
      
      entity->setRelCoor (relCoor);
      entity->setRelDir (relDir);
      entity->setMode(parentingMode);
      
      this->entities->add (entity);
      
      entity->postSpawn ();
    }
}



/**
  \brief commands that the world must catch
  \returns false if not used by the world
*/
bool World::command(Command* cmd)
{
  return false;
}

