
/*! 
    \file curve.h
    \brief A basic 3D curve framework
    
    Contains classes to handle curves
*/ 

#ifndef _CURVE_H
#define _CURVE_H

#include "vector.h"

//! An Enumerator that defines what sort of Curves are availible
enum CurveType {BEZIERCURVE, UPOINTCURVE};


//! An abstract class to handle curves. Needed for the Tracking system in orxonox.
class Curve
{
 protected:
  int nodeCount;         //!< The count of nodes the Curve has.
  Vector curvePoint;     //!< The point on the Cureve at a local Time.
  float localTime;       //!< If the time of one point is asked more than once the programm will not calculate it again.
  int derivation;        //!< Which derivation of a Curve is this.

  //! Handles the curve-points (dynamic List)
  struct PathNode
  {
    int number;          //!< The N-th node of this curve.
    float factor;        //!< Curve specific multiplier factor.
    Vector vFactor;      //!< A Vector-factor for multipliing.
    Vector position;     //!< Vector Pointung to this curve-point.
    PathNode* next;      //!< Pointer to the next Node.
  };

  PathNode* firstNode;   //!< First node of the curve.
  PathNode* currentNode; //!< The node we are working with (the Last node).


 private:
  virtual void rebuild(void) = 0;
 public:
  Curve(void);

  Curve* dirCurve;       //!< The derivation-curve of this Curve.
  void addNode(const Vector& newNode);
  void addNode(const Vector& newNode, unsigned int insertPosition);
  Vector getNode(unsigned int nodeToFind);
  inline int getNodeCount(void) { return this->nodeCount;}

  virtual Vector calcPos(float t) = 0;
  virtual Vector calcDir(float t) = 0;
  virtual Vector calcAcc(float t) = 0;
  virtual Quaternion calcQuat(float t) = 0;
 
  // DEBUG
  void debug(void);
};

//!    Class to handle bezier curves in 3-dimesnsional space
/**
   This Curve is good, for Fast Interaction. If you want to change it during the game, go on.
   !!be aware!! BezierCurves only flow through their first and last Node. Their Tangents at those Points a directed to the second and second-last Point.
*/
class BezierCurve : public Curve
{
 private:
  void rebuild(void);
 public:
  BezierCurve(void);
  BezierCurve(int derivation);
  virtual ~BezierCurve(void);

  Vector calcPos(float t);
  Vector calcDir(float t);
  Vector calcAcc(float t);
  Quaternion calcQuat(float t);
  
  
  Vector getPos(void) const;
};


//! B-Spline
/**
   class to handle b-spline in 3d space
*/
class BSplieCurve : public Curve
{


};

//! Uniform Point Curve-class
/**
   A UPoint Curve is a A Curve, that flows through all the nodes given it.
   The Algorithm to buid the curve is rather slow, but Painting and tracing along the curve has high speed, so do not change this curve during the Game.

   This Curve is very erattic, so i do not recommend to use it.
*/
class UPointCurve : public Curve
{
 private:
  void rebuild(void);
 public:
  UPointCurve(void);
  UPointCurve(int derivation);
  ~UPointCurve(void);

  Vector calcPos(float t);
  Vector calcDir(float t);
  Vector calcAcc(float t);
  Quaternion calcQuat(float t);
  
  Vector getPos(void) const;
};

#endif /* _CURVE_H */
