

/* 
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/


#include "light.h"

#include "glincl.h"

using namespace std;

//! Definition of the Lights
int lightsV[] = {GL_LIGHT0, GL_LIGHT1, GL_LIGHT2, GL_LIGHT3, GL_LIGHT4, GL_LIGHT5, GL_LIGHT6, GL_LIGHT7};


/**
   \brief standard constructor for a Light
*/
Light::Light () 
{
  this->setClassName ("Light");

  glEnable (GL_LIGHTING);
  this->setAmbientColor(.3, .3, .3);
  this->lights = new LightValue*[NUMBEROFLIGHTS];
  for (int i = 0; i < NUMBEROFLIGHTS; i++)
    lights[i] = NULL;
  this->currentLight = NULL;
}

/**
   \brief standard deconstructor
   
   first disables Lighting
   then deletes all the lights
   then deletes the rest of the allocated memory
   and in the end sets the singleton Reference to zero.
*/
Light::~Light () 
{
  this->destroy();
}


/**
   \brief frees all alocated memory

   and in this case also deletes the lightSources and GL_LIGHTING
*/
void Light::destroy(void)
{
  glDisable(GL_LIGHTING);
  
  for (int i = 0; i < NUMBEROFLIGHTS; i++)
    this->deleteLight(i);
  delete lights;
  Light::singletonRef = NULL;

  static_cast<WorldEntity*>(this)->destroy();
}


/**
   \brief singleton-Reference to the Light-class
*/
Light* Light::singletonRef = NULL;

/**
   \returns The Instance of the Lights
*/
Light* Light::getInstance(void)
{
  if (singletonRef)
    return singletonRef;
  else
    return Light::singletonRef = new Light();
}

/**
   \brief initializes a new Light with default values, and enables GL_LIGHTING
*/
void Light::init(int lightNumber)
{
  PRINTF(3)("initializing Light number %d.\n", lightNumber);
  // enable The light
  glEnable(lightsV[lightNumber]);
  this->currentLight = lights[lightNumber] = new LightValue;
  
  // set default values
  this->currentLight->lightNumber = lightNumber;
  this->setPosition(0.0, 0.0, 0.0);
  this->setDiffuseColor(1.0, 1.0, 1.0);
  this->setSpecularColor(1.0, 1.0, 1.0);
}

/**
   \brief Adds a new Light, by selecting the First free slot.

   if no slot is free error
*/
int Light::addLight(void)
{
  for (int i = 0; i < NUMBEROFLIGHTS; i++)
    if (!this->lights[i])
      return addLight(i); 
  PRINTF(1)("no more light slots availiable. All %d already taken\n", NUMBEROFLIGHTS);
  return -1;
}

/**
   \brief Adds a new Light
   \param lightNumber The number of the light to add.

   if the Number is not free: warn, automatically choose another slot.
*/
int Light::addLight(int lightNumber)
{
  if (this->lights[lightNumber])
    {
      PRINTF(2)("Lightslot %d is allready taken, trying another one\n", lightNumber);
      return this->addLight();
    }
  this->init(lightNumber);
  this->currentLight = this->lights[lightNumber];
  return lightNumber;
}

/**
   \brief select the light to work with
   \param lightNumber the light to work with
*/
void Light::editLightNumber(int lightNumber)
{
  if (!this->currentLight)
    { 
      PRINTF(1)("no Light defined yet\n");
      return;
    }

  this->currentLight = lights[lightNumber];
}

/**
   \brief Delete the current Light
*/
void Light::deleteLight(void)
{
  if (!this->currentLight)
    { 
      PRINTF(1)("no Light defined yet\n");
      return;
    }

  this->deleteLight(this->currentLight->lightNumber);
}

/**
   \brief delete light.
   \param lightNumber the number of the light to delete
*/
void Light::deleteLight(int lightNumber)
{
  if (this->lights[lightNumber])
    {
      PRINTF(3)("deleting Light number %d\n", lightNumber);
      delete this->lights[lightNumber];
      glDisable(lightsV[lightNumber]);
      this->lights[lightNumber] = NULL;
    }
}

// set Attributes
/**
   \brief Sets a Position for the Light.
   \param position The new position of the Light.
   \todo patrick: is it ok to set a Light Position even if it is derived from p_node??
*/
void Light::setPosition(Vector position)
{
  if (!this->currentLight)
    { 
      PRINTF(1)("no Light defined yet\n");
      return;
    }
  this->currentLight->lightPosition[0] = position.x;
  this->currentLight->lightPosition[1] = position.y;
  this->currentLight->lightPosition[2] = position.z;
  this->currentLight->lightPosition[3] = 0.0;

  glLightfv (GL_LIGHT0, GL_POSITION, this->currentLight->lightPosition);
}

/**
   \brief Sets a Position for the Light.
   \param x the x-coordinate
   \param y the y-coordinate
   \param z the z-coordinate
*/
void Light::setPosition(GLfloat x, GLfloat y, GLfloat z)
{
  if (!this->currentLight)
    { 
      PRINTF(1)("no Light defined yet\n");
      return;
    }

  this->currentLight->lightPosition[0] = x;
  this->currentLight->lightPosition[1] = y;
  this->currentLight->lightPosition[2] = z;
  this->currentLight->lightPosition[3] = 0.0;

  glLightfv (GL_LIGHT0, GL_POSITION, this->currentLight->lightPosition);
}

/**
   \brief sets an emitting Diffuse color for the Light
   \param r red
   \param g green
   \param b blue
*/
void Light::setDiffuseColor(GLfloat r, GLfloat g, GLfloat b)
{
  if (!this->currentLight)
    { 
      PRINTF(1)("no Light defined yet\n");
      return;
    }

  this->currentLight->diffuseColor[0] = r;
  this->currentLight->diffuseColor[1] = g;
  this->currentLight->diffuseColor[2] = b;
  this->currentLight->diffuseColor[3] = 1.0;

  glLightfv (GL_LIGHT0, GL_DIFFUSE, this->currentLight->diffuseColor);
}


/**
   \brief sets an emitting Ambient color for the Light
   \param r red
   \param g green
   \param b blue
*/
void Light::setSpecularColor(GLfloat r, GLfloat g, GLfloat b)
{
  if (!this->currentLight)
    { 
      PRINTF(1)("no Light defined yet\n");
      return;
    }

  this->currentLight->specularColor[0] = r;
  this->currentLight->specularColor[1] = g;
  this->currentLight->specularColor[2] = b;
  this->currentLight->specularColor[3] = 1.0;

  glLightfv (GL_LIGHT0, GL_SPECULAR, this->currentLight->specularColor);
}

/**
   \brief Sets the AttenuationType of this Light Source
   \param type the AttenuationType to set
   \param factor the Factor to multipy the attenuation with

   this actually just sets the following: glLightf(currentLight, type, factor)
*/
void Light::setAttenuation(AttenuationType type, float factor)
{
  if (!this->currentLight)
    { 
      PRINTF(1)("no Light defined yet\n");
      return;
    }
  this->currentLight->attenuationFactor = factor;
  this->currentLight->attenuationType = type;
  switch (type)
    {
    case CONSTANT:
      glLightf(lightsV[this->currentLight->lightNumber], GL_CONSTANT_ATTENUATION, factor);
      break;
    case LINEAR:
      glLightf(lightsV[this->currentLight->lightNumber], GL_LINEAR_ATTENUATION, factor);
      break;
    case QUADRATIC:
      glLightf(lightsV[this->currentLight->lightNumber], GL_QUADRATIC_ATTENUATION, factor);
      break;
    }
}


/**
   \brief sets the ambient Color of the Scene
   \param r red
   \param g green
   \param b blue
*/
void Light::setAmbientColor(GLfloat r, GLfloat g, GLfloat b)
{
  this->ambientColor[0] = r;
  this->ambientColor[1] = g;
  this->ambientColor[2] = b;
  this->ambientColor[3] = 1.0;

  glLightfv (GL_LIGHT0, GL_AMBIENT, this->ambientColor);
}

/**
   \brief stets the direction of the Spot Light.
   \param direction The direction of the Spot Light.
*/
void Light::setSpotDirection(Vector direction)
{
  this->currentLight->spotDirection[0] = direction.x;
  this->currentLight->spotDirection[1] = direction.y;
  this->currentLight->spotDirection[2] = direction.z;

  glLightfv(lightsV[this->currentLight->lightNumber], GL_SPOT_DIRECTION, this->currentLight->spotDirection);
}


/**
   \brief sets the cutoff angle of the Light.
   \param cutoff The cutoff angle.
*/
void Light::setSpotCutoff(GLfloat cutoff)
{
  this->currentLight->spotCutoff = cutoff;
  glLightf(lightsV[this->currentLight->lightNumber], GL_SPOT_CUTOFF, cutoff);
}


// get Attributes

/**
   \returns the Position of the Light
*/
Vector Light::getPosition(void)
{
  if (!this->currentLight)
    { 
      PRINTF(1)("no Light defined yet\n");
      return Vector(.0, .0, .0);
    }
  else
    return getPosition(currentLight->lightNumber);
}




/**
   \brief outputs debug information about the Class and its lights
*/
void Light::debug(void)
{
  PRINT(0)("=================================\n");
  PRINT(0)("= DEBUG INFORMATION CLASS LIGHT =\n");
  PRINT(0)("=================================\n");
  PRINT(0)("Reference: %p\n", Light::singletonRef);
  if (this->currentLight)
    PRINT(0)("current Light Nr: %d\n", this->currentLight->lightNumber);
  PRINT(0)("Ambient Color: %f:%f:%f\n", this->ambientColor[0], this->ambientColor[0], this->ambientColor[0]);
  PRINT(0)("=== Lights ===\n");
  for (int i = 0; i < NUMBEROFLIGHTS; i++)
    if (this->lights[i])
      {
	PRINT(0)(":: %d ::  -- reference %p\n", i, lights[i]);
	PRINT(0)(" GL-state: ");
	GLboolean param; 
	glGetBooleanv(lightsV[i], &param);
	if (param)
	  PRINT(0)("ON\n");
	else
	  PRINT(0)("OFF\n");

	if (i != lights[i]->lightNumber)
	  PRINTF(1)(" Lights are out of sync, this really should not happen,\n   %d % should be equal.\n", i, lights[i]->lightNumber);
	PRINT(0)(" Position:      %f/%f/%f\n", lights[i]->lightPosition[0], lights[i]->lightPosition[1], lights[i]->lightPosition[2]);
	PRINT(0)(" DiffuseColor:  %f/%f/%f\n", lights[i]->diffuseColor[0], lights[i]->diffuseColor[1], lights[i]->diffuseColor[2]);
	PRINT(0)(" SpecularColor: %f/%f/%f\n", lights[i]->specularColor[0], lights[i]->specularColor[1], lights[i]->specularColor[2]);
	PRINT(0)(" Attenuation:   ");
	switch (lights[i]->attenuationType)
	  {
	  case CONSTANT:
	    PRINT(0)("constant");
	  case LINEAR:
	    PRINT(0)("linear");
	    break;
	  case QUADRATIC:
	    PRINT(0)("quadratic");
	    break;
	  }
	PRINT(0)(" with Factor %f\n", lights[i]->attenuationFactor);
      }
  PRINT(0)("--------------------------------\n");
}
