
/* 
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Patrick Boenzli
   co-programmer: Christian Meyer
*/

#include "world.h"
#include "world_entity.h"
#include "track_manager.h"
#include "player.h"
#include "command_node.h"
#include "camera.h"
#include "environment.h"
#include "primitive.h"
#include "p_node.h"
#include "null_parent.h"
#include "helper_parent.h"
#include "glmenu_imagescreen.h"
#include "skysphere.h"
#include "light.h"
#include "fontset.h"

using namespace std;


/** 
    \brief create a new World
    
    This creates a new empty world!
*/
World::World (char* name)
{
  this->init(name, -1);
}

/**
   \brief creates a new World...
   \param worldID with this ID
*/
World::World (int worldID)
{
  this->init(NULL, worldID);
}

/** 
    \brief remove the World from memory
    
    delete everything explicitly, that isn't contained in the parenting tree!
    things contained in the tree are deleted automaticaly
*/
World::~World ()
{
  printf("World::~World() - deleting current world\n");
  CommandNode* cn = Orxonox::getInstance()->getLocalInput();
  cn->unbind(this->localPlayer);
  cn->reset();

  this->localCamera->destroy();
  this->nullParent->destroy();  
  //delete this->skySphere;

  //delete this->trackManager;

  /*
  WorldEntity* entity = entities->enumerate();  
  while( entity != NULL ) 
    { 
      entity->destroy();
      entity = entities->nextElement();
    }
  this->entities->destroy();
  */

  /* FIX the parent list has to be cleared - not possible if we got the old list also*/


  //delete this->entities;
  //delete this->localCamera;
  /* this->localPlayer hasn't to be deleted explicitly, it is 
     contained in entities*/
}

/**
   \brief initializes a new World
*/
void World::init(char* name, int worldID)
{
  this->setClassName ("World");

  this->worldName = name;
  this->debugWorldNr = worldID;
  this->entities = new tList<WorldEntity>();

  // Enable default GL stuff
  glEnable(GL_DEPTH_TEST);

}


/**
   \brief loads the World by initializing all resources, and set their default values.
*/
ErrorMessage World::load()
{
  //  BezierCurve* tmpCurve = new BezierCurve();
  if(this->debugWorldNr != -1)
    {
      // initializing Font
      testFont = new FontSet();
      testFont->buildFont("../data/pictures/font.tga");

      // initializing the TrackManager
      trackManager = TrackManager::getInstance();
      trackManager->addPoint(Vector(0,-5,0));
      trackManager->addPoint(Vector(10,0,5));
      trackManager->addPoint(Vector(20,0,-5));
      trackManager->addPoint(Vector(30,0,5));
      trackManager->addPoint(Vector(40,0,5));
      trackManager->setDuration(4);
      int fork11, fork12, fork13;
      trackManager->fork(3, &fork11, &fork12, &fork13);
      trackManager->workOn(fork11);
      trackManager->addPoint(Vector(70, 0, -10));
      trackManager->addPoint(Vector(100, 0, -15));
      trackManager->addPoint(Vector(300, 0, -15));
      trackManager->setDuration(10);
      trackManager->workOn(fork12);
      trackManager->addPoint(Vector(70,  0, 0));
      trackManager->addPoint(Vector(100, 0, 0));
      trackManager->addPoint(Vector(120,10, 0));
      trackManager->addPoint(Vector(150,10, 0));
      trackManager->addPoint(Vector(180,15, 0));
      trackManager->addPoint(Vector(200,10, 0));
      trackManager->setDuration(7);
      trackManager->workOn(fork13);
      trackManager->addPoint(Vector(70,  0, 10));
      trackManager->addPoint(Vector(100, 0, 30));
      trackManager->addPoint(Vector(120,-10, 30));
      trackManager->addPoint(Vector(150,-10, 30));
      trackManager->setDuration(10);
      trackManager->join(2, fork12, fork13);

      trackManager->workOn(5);
      trackManager->addPoint(Vector(250, 20, 10));
      trackManager->addPoint(Vector(290, 20, 10));
      trackManager->setDuration(5);
      trackManager->setSavePoint();
      trackManager->addPoint(Vector(350, 20, 10));
      trackManager->addPoint(Vector(360, 20, 10));
      trackManager->addPoint(Vector(370, 20, 10));
      trackManager->setDuration(5);

      trackManager->join(2, 6, fork11);

      trackManager->finalize();

      
      /*monitor progress*/
      this->glmis->step();

      /*
	tmpCurve->addNode(Vector(10,  -1,  -1));
	tmpCurve->addNode(Vector(10,  -2,   2));
	tmpCurve->addNode(Vector(10,   3,   3));
	tmpCurve->addNode(Vector(10,   4,  -4), 0);
	tmpCurve->addNode(Vector(10,  -1,  -1));
	tmpCurve->addNode(Vector(10,  -2,   2));
	tmpCurve->addNode(Vector(10,   3,   3));
	tmpCurve->addNode(Vector(10,   4,  -4), 0);
	tmpCurve->debug();
      */
      switch(this->debugWorldNr)
	{
	  /*
	    this loads the hard-coded debug world. this only for simplicity and will be 
	    removed by a reald world-loader, which interprets a world-file.
	    if you want to add an own debug world, just add a case DEBUG_WORLD_[nr] and
	    make whatever you want...
	   */
	case DEBUG_WORLD_0:
	  {
	    this->nullParent = NullParent::getInstance ();
	    this->nullParent->setName ("NullParent");

	    // !\todo old track-system has to be removed

	    //create helper for player
	    HelperParent* hp = new HelperParent ();
	    /* the player has to be added to this helper */

	    // create a player
	    WorldEntity* myPlayer = new Player ();
	    myPlayer->setName ("player");
	    this->spawn (myPlayer);
	    this->localPlayer = myPlayer;
	    /*monitor progress*/
	    this->glmis->step();	    

	    // bind input
	    Orxonox *orx = Orxonox::getInstance ();
	    orx->getLocalInput()->bind (myPlayer);
	    
	    // bind camera
	    this->localCamera = new Camera(this);
	    this->localCamera->setName ("camera");
	    this->localCamera->bind (myPlayer);
	    this->localPlayer->addChild (this->localCamera);
	    /*monitor progress*/
	    this->glmis->step();	    

	    // Create SkySphere
	    skySphere = new Skysphere("../data/pictures/sky-replace.jpg");
	    this->localPlayer->addChild(this->skySphere);

	    /*monitor progress*/
	    this->glmis->step();

	    
	    WorldEntity* env = new Environment();
	    env->setName ("env");
	    this->spawn(env);
	    this->trackManager->setBindSlave(env);

	    Vector* es = new Vector (50, 2, 0);
	    Quaternion* qs = new Quaternion ();
	    WorldEntity* pr = new Primitive(PSPHERE);
	    pr->setName("primitive");
	    this->spawn(pr, this->localPlayer, es, qs, ROTATION);
	    

	    /*monitor progress*/
	    this->glmis->step();

	    trackManager->setBindSlave(env);
	    trackManager->condition(1, NEAREST, myPlayer);

	    break;
	  }
	case DEBUG_WORLD_1:
	  {
	    /*
	    this->testCurve = new UPointCurve();
	    this->testCurve->addNode(Vector( 0, 0, 0));
	    this->testCurve->addNode(Vector(10, 0, 5));
	    this->testCurve->addNode(Vector(20, -5,-5));
	    this->testCurve->addNode(Vector(30, 5, 10));
	    this->testCurve->addNode(Vector(40, 0,-10));
	    this->testCurve->addNode(Vector(50, 0,-10));
	    */

	    this->nullParent = NullParent::getInstance ();
	    this->nullParent->setName ("NullParent");



	    // create a player
	    WorldEntity* myPlayer = new Player();
	    myPlayer->setName ("player");
	    this->spawn(myPlayer);
	    this->localPlayer = myPlayer;	    
	    
	    // bind input
	    Orxonox *orx = Orxonox::getInstance();
	    orx->getLocalInput()->bind (myPlayer);
	    
	    // bind camera
	    this->localCamera = new Camera (this);
	    this->localCamera->setName ("camera");
	    this->localCamera->bind (myPlayer);	
	    this->localPlayer->addChild (this->localCamera);

	    // Create SkySphere
	    skySphere = new Skysphere("../data/pictures/sky-replace.jpg");
	    this->localPlayer->addChild(this->skySphere);

	    break;


	  }
	default:
	  printf("World::load() - no world with ID %i found", this->debugWorldNr );
	}
    }
  else if(this->worldName != NULL)
    {

    }

  // initialize debug coord system
  objectList = glGenLists(1);
  glNewList (objectList, GL_COMPILE);

  glColor3f(1.0,0,0);

  int sizeX = 100;
  int sizeZ = 80;
  float length = 1000;
  float width = 200;
  float widthX = float (length /sizeX);
  float widthZ = float (width /sizeZ);
  
  float height [sizeX][sizeZ];
  Vector normal_vectors[sizeX][sizeZ];
  
  
  for ( int i = 0; i<sizeX-1; i+=1)
    for (int j = 0; j<sizeZ-1;j+=1)
      //height[i][j] = rand()/20046 + (j-25)*(j-25)/30;
#ifdef __WIN32__
      height[i][j]=(sin((float)j/3)*rand()*i/182400)*.5;
#else
      height[i][j]=(sin((float)j/3)*rand()*(long)i/6282450500.0)*.5;
#endif

  //Die Huegel ein wenig glaetten
  for (int h=1; h<2;h++)
    for (int i=1;i<sizeX-2 ;i+=1 )
      for(int j=1;j<sizeZ-2;j+=1)
	height[i][j]=(height[i+1][j]+height[i][j+1]+height[i-1][j]+height[i][j-1])/4;
  
  //Berechnung von normalen Vektoren
  for(int i=1;i<sizeX-2;i+=1)
    for(int j=1;j<sizeZ-2 ;j+=1)
      {
	Vector v1 = Vector (widthX*(1),      height[i][j],      widthZ*(j) );
	Vector v2 = Vector (widthX*(i-1),    height[i-1][j],    widthZ*(j));
	Vector v3 = Vector (widthX*(i),      height[i][j+1],    widthZ*(j+1));
	Vector v4 = Vector (widthX*(i+1),    height[i+1][j],    widthZ*(j));
	Vector v5 = Vector (widthX*(i),      height[i][j-1],    widthZ*(j-1));
	
	Vector c1 = v2 - v1;
	Vector c2 = v3 - v1;
	Vector c3=  v4 - v1;
	Vector c4 = v5 - v1;
	Vector zero = Vector (0,0,0);
	normal_vectors[i][j]=c1.cross(v3-v5)+c2.cross(v4-v2)+c3.cross(v5-v3)+c4.cross(v2-v4);
	normal_vectors[i][j].normalize();
      }

  glBegin(GL_QUADS);
  int snowheight=3;
  for ( int i = 0; i<sizeX; i+=1)
    for (int j = 0; j<sizeZ;j+=1)
      {	  
	Vector v1 = Vector (widthX*(i),      height[i][j]-20,       widthZ*(j)  -width/2);
	Vector v2 = Vector (widthX*(i+1),    height[i+1][j]-20,     widthZ*(j)  -width/2);
	Vector v3 = Vector (widthX*(i+1),    height[i+1][j+1]-20,   widthZ*(j+1)-width/2);
	Vector v4 = Vector (widthX*(i),      height[i][j+1]-20,     widthZ*(j+1)-width/2);
	float a[3];
	if(height[i][j]<snowheight){
	  a[0]=0;
	  a[1]=1.0-height[i][j]/10-.3;
	  a[2]=0;
	  glMaterialfv(GL_FRONT,GL_DIFFUSE,a);
	}
	else{
	    a[0]=1.0;
	    a[1]=1.0;
	    a[2]=1.0;
	    glMaterialfv(GL_FRONT,GL_DIFFUSE,a);
	    
	}
	glNormal3f(normal_vectors[i][j].x, normal_vectors[i][j].y, normal_vectors[i][j].z);
	glVertex3f(v1.x, v1.y, v1.z);
	if(height[i+1][j]<snowheight){
	  a[0]=0;
	  a[1] =1.0-height[i+1][j]/10-.3;
	  a[2]=0;
	  glMaterialfv(GL_FRONT,GL_DIFFUSE,a);
	}
	else{
	  a[0]=1.0;
	  a[1]=1.0;
	  a[2]=1.0;
	  glMaterialfv(GL_FRONT,GL_DIFFUSE,a);
	  
	}
	glNormal3f(normal_vectors[i+1][j].x, normal_vectors[i+1][j].y, normal_vectors[i+1][j].z);
	glVertex3f(v2.x, v2.y, v2.z);
	if(height[i+1][j+1]<snowheight){
	  a[0]=0;
	  a[1] =1.0-height[i+1][j+1]/10-.3;
	  a[2]=0;
	  glMaterialfv(GL_FRONT,GL_DIFFUSE,a);
	}
	else{
	  a[0]=1.0;
	  a[1]=1.0;
	  a[2]=1.0;
	  glMaterialfv(GL_FRONT,GL_DIFFUSE,a);
	  
	  
	}
	glNormal3f(normal_vectors[i+1][j+1].x, normal_vectors[i+1][j+1].y, normal_vectors[i+1][j+1].z);
	glVertex3f(v3.x, v3.y, v3.z);
	if(height[i][j+1]<snowheight){
	  a[0]=0;
	  a[1] =1.0-height[i+1][j+1]/10-.3;
	  a[2]=0;
	  glMaterialfv(GL_FRONT,GL_DIFFUSE,a);
	}
	else{
	  a[0]=1.0;
	  a[1]=1.0;
	  a[2]=1.0;
	  glMaterialfv(GL_FRONT,GL_DIFFUSE,a);
	}
	glNormal3f(normal_vectors[i][j+1].x, normal_vectors[i][j+1].y, normal_vectors[i][j+1].z);
	glVertex3f(v4.x, v4.y, v4.z);
	
      }
  glEnd();

  trackManager->drawGraph(.01);
  trackManager->debug(2);
  glEndList();


  // LIGHT initialisation
  light = Light::getInstance();
  light->addLight(0);
  light->setAttenuation(QUADRATIC, 1.0);
  light->setAttenuation(CONSTANT, 2.0);
  light->setAttenuation(QUADRATIC, 1.0);
  light->setPosition(10.0, 10.0, 50.0);
  light->setDiffuseColor(1,1,1);
  //  light->addLight(1);
  //  light->setPosition(20, 10, -20);
  //  light->setDiffuseColor(0,0,0);
  light->debug();


}

/**
   \brief initializes a new World
*/
ErrorMessage World::init()
{
  this->bPause = false;
  CommandNode* cn = Orxonox::getInstance()->getLocalInput();
  cn->addToWorld(this);
  cn->enable(true);

  //glMap1f (GL_MAP1_VERTEX_3, 0.0, 1.0, 3, 4, &ctrlpoints[0][0]);
  //glEnable (GL_MAP1_VERTEX_3);
  
  //theNurb = gluNewNurbsRenderer ();
  //gluNurbsProperty (theNurb, GLU_NURBS_MODE, GLU_NURBS_TESSELLATOR);
  //gluNurbsProperty (theNurb, GLU_NURBS_VERTEX, vertexCallback );

}


/**
   \brief starts the World
*/
ErrorMessage World::start()
{
  printf("World::start() - starting current World: nr %i\n", this->debugWorldNr);
  this->bQuitOrxonox = false;
  this->bQuitCurrentGame = false;
  this->mainLoop();
}

/**
   \brief stops the world.

   This happens, when the player decides to end the Level.
*/
ErrorMessage World::stop()
{
  printf("World::stop() - got stop signal\n");
  this->bQuitCurrentGame = true;
}

/**
   \brief pauses the Game
*/
ErrorMessage World::pause()
{
  this->isPaused = true;
}

/**
   \brief ends the pause Phase
*/
ErrorMessage World::resume()
{
  this->isPaused = false;
}

/**
   \brief destroys the World
*/
ErrorMessage World::destroy()
{
  delete trackManager;
}

/**
   \brief shows the loading screen
*/
void World::displayLoadScreen ()
{
  printf ("World::displayLoadScreen - start\n"); 
  
  //GLMenuImageScreen* 
  this->glmis = GLMenuImageScreen::getInstance();
  this->glmis->init();
  this->glmis->setMaximum(10);
  this->glmis->draw();
 
  printf ("World::displayLoadScreen - end\n"); 
}

/**
   \brief removes the loadscreen, and changes over to the game

   \todo take out the delay
*/
void World::releaseLoadScreen ()
{
  printf ("World::releaseLoadScreen - start\n"); 
  this->glmis->setValue(this->glmis->getMaximum());
  SDL_Delay(500);
  printf ("World::releaseLoadScreen - end\n"); 
}


/** 
    \brief checks for collisions
    
    This method runs through all WorldEntities known to the world and checks for collisions 
    between them. In case of collisions the collide() method of the corresponding entities 
    is called.
*/
void World::collide ()
{
  /*
  List *a, *b;
  WorldEntity *aobj, *bobj;
   
  a = entities;
  
  while( a != NULL)
    {
      aobj = a->nextElement();
      if( aobj->bCollide && aobj->collisioncluster != NULL)
	{
	  b = a->nextElement();
	  while( b != NULL )
	    {
	      bobj = b->nextElement();
	      if( bobj->bCollide && bobj->collisioncluster != NULL )
		{
		  unsigned long ahitflg, bhitflg;
		  if( check_collision ( &aobj->place, aobj->collisioncluster, 
					&ahitflg, &bobj->place, bobj->collisioncluster, 
					&bhitflg) );
		  {
		    aobj->collide (bobj, ahitflg, bhitflg);
		    bobj->collide (aobj, bhitflg, ahitflg);
		  }
		}
	      b = b->nextElement();
	    }
	}
      a = a->enumerate();
    }
  */
}

/** 
    \brief runs through all entities calling their draw() methods
*/
void World::draw ()
{
  /* draw entities */
  WorldEntity* entity;
  glLoadIdentity();

  entity = this->entities->enumerate();
  while( entity != NULL ) 
    { 
      if( entity->bDraw ) entity->draw();
      entity = this->entities->nextElement();
    } 
  
  glCallList (objectList);
  //! \todo skysphere is a WorldEntity and should be inside of the world-entity-list.
  skySphere->draw();

  testFont->printText(0, 0, 1, "orxonox_" PACKAGE_VERSION);

}


/**
   \brief function to put your own debug stuff into it. it can display informations about
   the current class/procedure
*/
void World::debug()
{
  printf ("World::debug() - starting debug\n");
  PNode* p1 = NullParent::getInstance ();
  PNode* p2 = new PNode (new Vector(2, 2, 2), p1);
  PNode* p3 = new PNode (new Vector(4, 4, 4), p1);
  PNode* p4 = new PNode (new Vector(6, 6, 6), p2);

  p1->debug ();
  p2->debug ();
  p3->debug ();
  p4->debug ();

  p1->shiftCoor (new Vector(-1, -1, -1));

  printf("World::debug() - shift\n");
  p1->debug ();
  p2->debug ();
  p3->debug ();
  p4->debug ();
  
  p1->update (1);

  printf ("World::debug() - update\n");
  p1->debug ();
  p2->debug ();
  p3->debug ();
  p4->debug ();

  p2->shiftCoor (new Vector(-1, -1, -1));
  p1->update (2);

  p1->debug ();
  p2->debug ();
  p3->debug ();
  p4->debug ();

  p2->setAbsCoor (new Vector(1,2,3));


 p1->update (2);

  p1->debug ();
  p2->debug ();
  p3->debug ();
  p4->debug ();

  p1->destroy ();
  
  
  /*
  WorldEntity* entity;
  printf("counting all entities\n");
  printf("World::debug() - enumerate()\n");
  entity = entities->enumerate();  
  while( entity != NULL ) 
    { 
      if( entity->bDraw ) printf("got an entity\n");
      entity = entities->nextElement();
    }
  */
}


/**
  \brief main loop of the world: executing all world relevant function

  in this loop we synchronize (if networked), handle input events, give the heart-beat to
  all other member-entities of the world (tick to player, enemies etc.), checking for
  collisions drawing everything to the screen.
*/
void World::mainLoop()
{
  this->lastFrame = SDL_GetTicks ();
  printf("World::mainLoop() - Entering main loop\n");
  while( !this->bQuitOrxonox && !this->bQuitCurrentGame) /* \todo implement pause */
    {
      // Network
      this->synchronize ();
      // Process input
      this->handleInput ();
      if( this->bQuitCurrentGame || this->bQuitOrxonox)
	{
	  printf("World::mainLoop() - leaving loop earlier...\n");
	  break;
	}
      // Process time
      this->timeSlice ();
      // Process collision
      this->collide ();
      // Draw
      this->display ();
  
      for( int i = 0; i < 5000000; i++) {}
      /* \todo this is to slow down the program for openGl Software emulator computers, reimplement*/
    }
  printf("World::mainLoop() - Exiting the main loop\n");
}


/**
   \brief synchronize local data with remote data
*/
void World::synchronize ()
{
  // Get remote input
  // Update synchronizables
}


/**
   \brief run all input processing

   the command node is the central input event dispatcher. the node uses the even-queue from
   sdl and has its own event-passing-queue.
*/
void World::handleInput ()
{
  // localinput
  CommandNode* cn = Orxonox::getInstance()->getLocalInput();
  cn->process();
  // remoteinput
}


/**
   \brief advance the timeline

   this calculates the time used to process one frame (with all input handling, drawing, etc)
   the time is mesured in ms and passed to all world-entities and other classes that need
   a heart-beat.
*/
void World::timeSlice ()
{
  Uint32 currentFrame = SDL_GetTicks();
  if(!this->bPause)
    {
      Uint32 dt = currentFrame - this->lastFrame;
      
      if(dt > 0)
	{
	  float fps = 1000/dt;
	  printf("fps = %f\n", fps);
	}
      else
	{
	  /* the frame-rate is limited to 100 frames per second, all other things are for
	     nothing.
	  */
	  printf("fps = 1000 - frame rate is adjusted\n");
	  SDL_Delay(10);
	  dt = 10;
	}
      //this->timeSlice (dt);
      
      /* function to let all entities tick (iterate through list) */
      WorldEntity* entity;
      float seconds = dt / 1000.0;      
      this->nullParent->update (seconds);
      entity = entities->enumerate(); 
      while( entity != NULL) 
	{ 
	  entity->tick (seconds);
	  entity = entities->nextElement();
	}
      //skySphere->updatePosition(localCamera->absCoordinate);
      
      /* update tick the rest */
      this->localCamera->timeSlice(dt);
      this->trackManager->tick(dt);
    }
  this->lastFrame = currentFrame;
}


/**
   \brief render the current frame
   
   clear all buffers and draw the world
*/
void World::display ()
{
  // clear buffer
  glClear( GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);
  // set camera
  this->localCamera->apply ();
  // draw world
  this->draw();
  // draw HUD
  /* \todo draw HUD */
  // flip buffers
  SDL_GL_SwapBuffers();
  //SDL_Surface* screen = Orxonox::getInstance()->getScreen ();
  //SDL_Flip (screen);
}


/**
   \brief add and spawn a new entity to this world
   \param entity to be added
*/
void World::spawn(WorldEntity* entity)
{
  if( this->nullParent != NULL && entity->parent == NULL)
    this->nullParent->addChild (entity);

  this->entities->add (entity);

  entity->postSpawn ();
}


/**
   \brief add and spawn a new entity to this world
   \param entity to be added
   \param absCoor At what coordinates to add this entity.
   \param absDir In which direction should it look.
*/
void World::spawn(WorldEntity* entity, Vector* absCoor, Quaternion* absDir)
{
  entity->setAbsCoor (absCoor);
  entity->setAbsDir (absDir);
  
  if( this->nullParent != NULL && entity->parent == NULL)
    this->nullParent->addChild (entity);

  this->entities->add (entity);

  entity->postSpawn ();
}


/**
   \brief add and spawn a new entity to this world
   \param entity to be added
   \param entity to be added to (PNode)
   \param At what relative  coordinates to add this entity.
   \param In which relative direction should it look.
*/
void World::spawn(WorldEntity* entity, PNode* parentNode, 
		  Vector* relCoor, Quaternion* relDir, 
		  parentingMode mode)
{

  if( parentNode != NULL && entity->parent == NULL)
    {
      parentNode->addChild (entity);
      
      entity->setRelCoor (relCoor);
      entity->setRelDir (relDir);
      
      this->entities->add (entity);
      
      entity->postSpawn ();
    }
}



/**
  \brief commands that the world must catch
  \returns false if not used by the world
*/
bool World::command(Command* cmd)
{
  return false;
}

