/* 
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  


   ### File Specific:
   main-programmer: Patrick Boenzli
   co-programmer: Christian Meyer
*/

#include "orxonox.h"
#include "world.h"
#include "camera.h"
#include "data_tank.h"
#include "command_node.h"
#include "game_loader.h"
#include <string.h>
int verbose;

using namespace std;

/**
   \brief create a new Orxonox
*/
Orxonox::Orxonox ()
{
  pause = false;
}

/**
   \brief remove Orxonox from memory
*/
Orxonox::~Orxonox () 
{
  Orxonox::singletonRef = NULL;
  if( world != NULL) delete world;
  if( localinput != NULL) delete world;
  if( localcamera != NULL) delete localcamera;
  if( resources != NULL) delete resources;
}


/** \brief this is a singleton class to prevent duplicates */
Orxonox* Orxonox::singletonRef = 0;

/**
   \returns reference or new Object of Orxonox if not existent.
*/
Orxonox* Orxonox::getInstance (void)
{
  if (singletonRef == NULL)
    singletonRef = new Orxonox();
  return singletonRef;
}

/**
   \brief this finds the config file
   
   Since the config file varies from user to user and since one may want to specify different config files
   for certain occasions or platforms this function finds the right config file for every occasion and stores
   it's path and name into configfilename
*/
void Orxonox::getConfigFile (int argc, char** argv)
{
  strcpy (configfilename, "orxonox.conf");
}

/**
   \brief initialize Orxonox with command line
*/
int Orxonox::init (int argc, char** argv)
{
  // parse command line
  // config file
  
  getConfigFile (argc, argv);
  SDL_Init (SDL_INIT_TIMER);
  // initialize everything
  if( initVideo() == -1) return -1;
  if( initSound() == -1) return -1;
  printf("> Initializing input\n");
  if( initInput() == -1) return -1;
  printf("> Initializing networking\n");
  if( initNetworking () == -1) return -1;
  printf("> Initializing resources\n");
  if( initResources () == -1) return -1;
  //printf("> Initializing world\n");
  //if( init_world () == -1) return -1; PB: world will be initialized when started
  
  return 0;
}

/**
   \brief initializes SDL and OpenGL
*/
int Orxonox::initVideo() 
{
  printf("> Initializing video\n");
  if (SDL_Init(SDL_INIT_VIDEO) == -1)
    {
      printf ("could not initialize SDL Video\n");
      return -1;
    }
  // Set video mode
  // TO DO: parse arguments for settings
  //SDL_GL_SetAttribute(SDL_GL_RED_SIZE, 5);
  //SDL_GL_SetAttribute(SDL_GL_GREEN_SIZE, 5);
  //SDL_GL_SetAttribute(SDL_GL_BLUE_SIZE, 5);
  //SDL_GL_SetAttribute(SDL_GL_DEPTH_SIZE, 16);
  

  SDL_GL_SetAttribute( SDL_GL_DOUBLEBUFFER, 1 );   
  SDL_GL_SetAttribute( SDL_GL_DEPTH_SIZE, 16);    
  SDL_GL_SetAttribute( SDL_GL_STENCIL_SIZE, 0);  
  SDL_GL_SetAttribute( SDL_GL_ACCUM_RED_SIZE, 0);
  SDL_GL_SetAttribute( SDL_GL_ACCUM_GREEN_SIZE, 0);
  SDL_GL_SetAttribute( SDL_GL_ACCUM_BLUE_SIZE, 0);
  SDL_GL_SetAttribute( SDL_GL_ACCUM_ALPHA_SIZE, 0);



  int bpp = 16;
  int width = 640;
  int height = 480;
  //Uint32 flags = SDL_HWSURFACE | SDL_OPENGL | SDL_GL_DOUBLEBUFFER; /* \todo: SDL_OPENGL doen't permit to load images*/
  //Uint32 flags = SDL_HWSURFACE | SDL_GL_DOUBLEBUFFER;

  Uint32 videoFlags = SDL_OPENGL | SDL_HWPALETTE | SDL_RESIZABLE;

  /* query SDL for information about our video hardware */
  const SDL_VideoInfo* videoInfo = SDL_GetVideoInfo ();
  
  if( videoInfo == NULL)
    {
      printf ("Orxonox::initVideo() - Failed getting Video Info :%s\n", SDL_GetError()); 
      SDL_Quit ();
    }
  if( videoInfo->hw_available)
    videoFlags |= SDL_HWSURFACE;
  else 
    videoFlags |= SDL_SWSURFACE;
  /*
  if(VideoInfo -> blit_hw)                            
    VideoFlags |= SDL_HWACCEL;
  */
  
  if((this->screen = SDL_SetVideoMode (width, height, bpp, videoFlags)) == NULL)
  {
    printf("Could not SDL_SetVideoMode(%d, %d, %d, %d): %s\n", width, height, bpp, videoFlags, SDL_GetError());
    SDL_Quit();
    return -1;
  }
  
  // Set window labeling
  SDL_WM_SetCaption ("Orxonox " PACKAGE_VERSION, "Orxonox " PACKAGE_VERSION);
  
  // TO DO: Create a cool icon and use it here
  // SDL_WM_SetIcon(SDL_Surface *icon, Uint8 *mask);  

  return 0;
}


/**
   \brief initializes the sound engine
*/
int Orxonox::initSound() 
{
  printf("> Initializing sound\n");
  // SDL_Init(SDL_INIT_AUDIO);
  printf("Not yet implemented\n");
  return 0;
}


/**
   \brief initializes input functions
*/
int Orxonox::initInput() 
{
  // create localinput
  localinput = new CommandNode( configfilename);
  
  return 0;
}


/**
   \brief initializes network system
*/
int Orxonox::initNetworking() 
{
  printf("Not yet implemented\n");
  return 0;
}


/**
   \brief initializes and loads resource files
*/
int Orxonox::initResources() 
{
  printf("Not yet implemented\n");
  return 0;
}


/**
   \brief initializes the world
*/
int Orxonox::initWorld() 
{
  //world = new World();
  
  // TO DO: replace this with a menu/intro
  //world->load_debug_level();
  
  return 0;
}


/**
   \brief starts the orxonox game or menu

   here is the central orxonox state manager. There are currently two states
   - menu
   - game-play
   both states manage their states themselfs again.
*/
void Orxonox::start()
{
  
  this->gameLoader = GameLoader::getInstance();
  this->gameLoader->loadDebugCampaign(DEBUG_CAMPAIGN_0);
  this->gameLoader->init();
  this->gameLoader->start();
}


/**
   \brief exits Orxonox
*/
void Orxonox::quitGame() 
{
  bQuitOrxonox = true;
}



/**
   \brief handles sprecial events from localinput
   \param event: an event not handled by the CommandNode 
*/
void Orxonox::eventHandler(SDL_Event* event)
{
  // Handle special events such as reshape, quit, focus changes
}
 

/**
   \brief handle keyboard commands that are not meant for WorldEntities
   \param cmd: the command to handle
   \return true if the command was handled by the system or false if it may be passed to the WorldEntities
*/
bool Orxonox::systemCommand(Command* cmd)
{
  /*
  if( !strcmp( cmd->cmd, "quit"))
    {
      if( !cmd->bUp) this->gameLoader->stop();
      return true;
    }
  return false;
  */
  return false;
}


/**
   \brief retrieve a pointer to the local Camera
   \return a pointer to localcamera
*/
Camera* Orxonox::getCamera()
{
  return localcamera;
}


/**
   \brief retrieve a pointer to the local CommandNode
   \return a pointer to localinput
*/
CommandNode* Orxonox::getLocalInput()
{
  return localinput;
}


/**
   \brief retrieve a pointer to the local World
   \return a pointer to world
*/
World* Orxonox::getWorld()
{
  return world;
}

/**
   \return The reference of the SDL-screen of orxonox
*/
SDL_Surface* Orxonox::getScreen ()
{
  return this->screen;
}

/**
   \brief main function

   here the journey begins
*/
int main(int argc, char** argv) 
{  
  printf(">>> Starting Orxonox <<<\n");
  Orxonox *orx = Orxonox::getInstance();
  
  if((*orx).init(argc, argv) == -1)
    {
      printf("! Orxonox initialization failed\n");
      return -1;
    }
  
  orx->start();
  
  //delete orx;
  
  return 0;
}
