
/* 
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Patrick Boenzli
   co-programmer: Christian Meyer
*/

#include "world.h"
#include "world_entity.h"
#include "track_manager.h"
#include "player.h"
#include "command_node.h"
#include "camera.h"
#include "environment.h"
#include "p_node.h"
#include "null_parent.h"
#include "helper_parent.h"
#include "glmenu_imagescreen.h"
#include "skysphere.h"
#include "light.h"
#include "fontset.h"
#include "factory.h"

using namespace std;


World::World( TiXmlElement* root)
{
	char *string, *name;
	int id;

	// identifier
	string = grabParameter( root, "identifier");
	if( string == NULL || if( sscanf(string, "%d", &id) != 1))
	{
		PRINTF(ERR)("World is missing a proper 'identifier'\n");
		setStoryId( -1);
	}
	else setStoryId( id);
	
	// path
	string = grabParameter( root, "path");
	if( string == NULL)
	{
		PRINTF(ERR)("World is missing a proper 'path'\n");
		setPath( NULL);
	}
	else
	{
		name = new char[strlen(string + 2)];
		strcpy( name, string);
		setPath( name);
	}
	
	
}

/** 
    \brief create a new World
    
    This creates a new empty world!
*/
World::World (char* name)
{
  this->setClassName ("World");
  this->worldName = name;
  this->debugWorldNr = -1;
  this->entities = new tList<WorldEntity>();
}

/**
   \brief creates a new World...
   \param worldID with this ID
*/
World::World (int worldID)
{
  this->debugWorldNr = worldID;
  this->worldName = NULL;
  this->entities = new tList<WorldEntity>();
}

/** 
    \brief remove the World from memory
    
    delete everything explicitly, that isn't contained in the parenting tree!
    things contained in the tree are deleted automaticaly
*/
World::~World ()
{
  printf("World::~World() - deleting current world\n");
  CommandNode* cn = Orxonox::getInstance()->getLocalInput();
  cn->unbind(this->localPlayer);
  cn->reset();

  this->localCamera->destroy();
  this->nullParent->destroy();  
  delete this->skySphere;

  //delete this->trackManager;

  /*
  WorldEntity* entity = entities->enumerate();  
  while( entity != NULL ) 
    { 
      entity->destroy();
      entity = entities->nextElement();
    }
  this->entities->destroy();
  */

  /* FIX the parent list has to be cleared - not possible if we got the old list also*/


  //delete this->entities;
  //delete this->localCamera;
  /* this->localPlayer hasn't to be deleted explicitly, it is 
     contained in entities*/
}


/**
   \brief loads the World by initializing all resources, and set their default values.
*/
ErrorMessage World::load()
{
	GameLoader* loader = GameLoader::getInstance();
	
  if( getPath() == NULL)
  {
		PRINTF(ERR)("World has no path specified for loading");
		return (ErrorMessage){213,"Path not specified","World::load()"};
  }
  
	TiXmlDocument* XMLDoc = new TiXmlDocument( name);
	// load the campaign document
	if( !XMLDoc.LoadFile())
	{
		// report an error
		PRINTF(ERR)("Error loading XML File: %s @ %d:%d\n", XMLDoc.ErrorDesc(), XMLDoc.ErrorRow(), XMLDoc.ErrorCol());
		delete XMLDoc;
		return (ErrorMessage){213,"XML File parsing error","World::load()"};
	}
	
	// check basic validity
	TiXmlElement* root = XMLDoc.RootElement();
	assert( root != NULL);
	
	element = root->FirstChildElement( "WorldDataFile");
	
	if( root == NULL )
	{
		// report an error
		PRINTF(ERR)("Specified XML File is not an orxonox world data file (WorldDataFile element missing)\n");
		delete XMLDoc;
		return (ErrorMessage){213,"Path not a WorldDataFile","World::load()"};
	}
	
	// load the parameters
		// name
	TiXmlElement* element;
	char* temp;
	char* string = grabParameter( root, "name");
	if( string == NULL)
	{
		PRINTF(ERR)("World is missing a proper 'name'\n");
		setStoryId( -1);
	}
	else
	{
		temp = new char[strlen(string + 2)];
		worldName = temp;
	}
	
	
	// find WorldEntities
  element = root->FirstChildElement( "WorldEntities");
  
  if( element == NULL)
  {
		PRINTF(ERR)("World is missing 'WorldEntities'\n");
  }
  else
  {
  	element = element->FirstChildElement();
	  // load Players/Objects/Whatever
		while( element != NULL)
		{
			WorldEntity* created = (WorldEntity*) loader->fabricate( element);
			if( created != NULL) spawn( created);
			element = element->nextSiblingElement();
		}
	}
	
	// find Track
  element = root->FirstChildElement( "Track");
  if( element == NULL)
  {
		PRINTF(ERR)("World is missing a 'Track'\n");
  }
  else
  {	
  	//load track
  	trackManager = TrackManager::getInstance();
  	trackManager->loadTrack( element);
	}
  
	
	// free the XML data
	delete XMLDoc;
	
  //  BezierCurve* tmpCurve = new BezierCurve();
  if(this->debugWorldNr != -1)
    {
      // initializing Font
      testFont = new FontSet();
      testFont->buildFont("../data/pictures/font.tga");

      // initializing the TrackManager
      trackManager = TrackManager::getInstance();
      trackManager->addPoint(Vector(0,-5,0));
      trackManager->addPoint(Vector(10,0,5));
      trackManager->addPoint(Vector(20,0,-5));
      trackManager->addPoint(Vector(30,0,5));
      trackManager->addPoint(Vector(40,0,5));
      trackManager->setDuration(2);
      trackManager->setSavePoint();
      trackManager->addPoint(Vector(50,10,10));
      trackManager->addPoint(Vector(60,0, 10));
      trackManager->addPoint(Vector(70,0, 10));
      trackManager->addPoint(Vector(80,0,-10));
      trackManager->addPoint(Vector(90,0,-10));
      trackManager->setDuration(5);
      trackManager->setSavePoint();
      trackManager->addPoint(Vector(110,0,5));
      trackManager->addPoint(Vector(120,0, 10));
      trackManager->addPoint(Vector(130,0, 10));
      trackManager->addPoint(Vector(140,0,-10));
      trackManager->addPoint(Vector(150,0,-10));
      trackManager->setDuration(3);
      int fork11, fork12, fork13, fork14;
      trackManager->fork(4, &fork11, &fork12, &fork13, &fork14);
      trackManager->workOn(fork11);
      trackManager->addPoint(Vector(170, 0, -15));
      trackManager->addPoint(Vector(180, 0, -15));
      trackManager->setDuration(3);
      trackManager->workOn(fork12);
      trackManager->addPoint(Vector(170, 0, 10));
      trackManager->addPoint(Vector(180, 0, 10));
      trackManager->addPoint(Vector(190,2,5));
      trackManager->addPoint(Vector(200,2,5));
      trackManager->setDuration(7);
      int fork21, fork22;
      trackManager->fork(2, &fork21, &fork22);
      trackManager->workOn(fork21);
      trackManager->addPoint(Vector(220, 10,-10));
      trackManager->addPoint(Vector(230, 0,-10));
      trackManager->addPoint(Vector(240, 0, 2));
      trackManager->addPoint(Vector(250, 0, 0));
      trackManager->addPoint(Vector(260, 0, 5));
      trackManager->setDuration(3);
      trackManager->join(2, fork12, fork11);
      trackManager->workOn(fork22);
      trackManager->addPoint(Vector(220, -10,10));
      trackManager->addPoint(Vector(230, 0, 10));
      trackManager->addPoint(Vector(240, 0, 10));
      trackManager->addPoint(Vector(250, 0, 5));
      trackManager->setDuration(6);
      trackManager->workOn(fork13);
      trackManager->addPoint(Vector(200,-10,5));
      trackManager->addPoint(Vector(250,-10,5));
      trackManager->setDuration(3);
      trackManager->workOn(fork14);
      trackManager->addPoint(Vector(200,15,0));
      trackManager->addPoint(Vector(210,0,10));
      trackManager->setDuration(1);
      trackManager->join(4, fork21, fork22, fork13, fork14);
      trackManager->workOn(10);
      trackManager->addPoint(Vector(250,-10,5));
      trackManager->addPoint(Vector(260,-10,5));
      trackManager->finalize();
      
      /*monitor progress*/
      this->glmis->step();

      /*
	tmpCurve->addNode(Vector(10,  -1,  -1));
	tmpCurve->addNode(Vector(10,  -2,   2));
	tmpCurve->addNode(Vector(10,   3,   3));
	tmpCurve->addNode(Vector(10,   4,  -4), 0);
	tmpCurve->addNode(Vector(10,  -1,  -1));
	tmpCurve->addNode(Vector(10,  -2,   2));
	tmpCurve->addNode(Vector(10,   3,   3));
	tmpCurve->addNode(Vector(10,   4,  -4), 0);
	tmpCurve->debug();
      */
      switch(this->debugWorldNr)
	{
	  /*
	    this loads the hard-coded debug world. this only for simplicity and will be 
	    removed by a reald world-loader, which interprets a world-file.
	    if you want to add an own debug world, just add a case DEBUG_WORLD_[nr] and
	    make whatever you want...
	   */
	case DEBUG_WORLD_0:
	  {
	    this->nullParent = NullParent::getInstance ();
	    this->nullParent->setName ("NullParent");

	    // !\todo old track-system has to be removed

	    //create helper for player
	    HelperParent* hp = new HelperParent ();
	    /* the player has to be added to this helper */

	    // create a player
	    WorldEntity* myPlayer = new Player ();
	    myPlayer->setName ("player");
	    this->spawn (myPlayer);
	    this->localPlayer = myPlayer;
	    /*monitor progress*/
	    this->glmis->step();	    

	    // bind input
	    Orxonox *orx = Orxonox::getInstance ();
	    orx->getLocalInput()->bind (myPlayer);
	    
	    // bind camera
	    this->localCamera = new Camera(this);
	    this->localCamera->setName ("camera");
	    this->localCamera->bind (myPlayer);
	    this->localPlayer->addChild (this->localCamera);

	    // Create SkySphere
	    skySphere = new Skysphere("../data/pictures/sky-replace.jpg");

	    /*monitor progress*/
	    this->glmis->step();

	    Vector* es = new Vector (50, 2, 0);
	    Quaternion* qs = new Quaternion ();
	    WorldEntity* env = new Environment();
	    env->setName ("env");
	    this->spawn(env, es, qs);
	   
	    /*monitor progress*/
	    this->glmis->step();

	    trackManager->setBindSlave(env);

	    break;
	  }
	case DEBUG_WORLD_1:
	  {
	    /*
	    this->testCurve = new UPointCurve();
	    this->testCurve->addNode(Vector( 0, 0, 0));
	    this->testCurve->addNode(Vector(10, 0, 5));
	    this->testCurve->addNode(Vector(20, -5,-5));
	    this->testCurve->addNode(Vector(30, 5, 10));
	    this->testCurve->addNode(Vector(40, 0,-10));
	    this->testCurve->addNode(Vector(50, 0,-10));
	    */

	    this->nullParent = NullParent::getInstance ();
	    this->nullParent->setName ("NullParent");



	    // create a player
	    WorldEntity* myPlayer = new Player();
	    myPlayer->setName ("player");
	    this->spawn(myPlayer);
	    this->localPlayer = myPlayer;	    
	    
	    // bind input
	    Orxonox *orx = Orxonox::getInstance();
	    orx->getLocalInput()->bind (myPlayer);
	    
	    // bind camera
	    this->localCamera = new Camera (this);
	    this->localCamera->setName ("camera");
	    this->localCamera->bind (myPlayer);	
	    this->localPlayer->addChild (this->localCamera);

	    // Create SkySphere
	    skySphere = new Skysphere("../data/pictures/sky-replace.jpg");

	    break;


	  }
	default:
	  printf("World::load() - no world with ID %i found", this->debugWorldNr );
	}
    }
  else if(this->worldName != NULL)
    {

    }

  // initialize debug coord system
  objectList = glGenLists(1);
  glNewList (objectList, GL_COMPILE);
  glLoadIdentity();
  glColor3f(1.0,0,0);
  glBegin(GL_QUADS);

  int sizeX = 100;
  int sizeZ = 80;
  float length = 1000;
  float width = 200;
  float widthX = float (length /sizeX);
  float widthZ = float (width /sizeZ);
  
  float height [sizeX][sizeZ];
  Vector normal_vectors[sizeX][sizeZ];
  
  
  for ( int i = 0; i<sizeX-1; i+=1)
    for (int j = 0; j<sizeZ-1;j+=1)
      //height[i][j] = rand()/20046 + (j-25)*(j-25)/30;
#ifdef __WIN32__
      height[i][j]=(sin((float)j/3)*rand()*i/182400)*.5;
#else
      height[i][j]=(sin((float)j/3)*rand()*(long)i/6282450500.0)*.5;
#endif

  //Die Huegel ein wenig glaetten
  for (int h=1; h<2;h++)
    for (int i=1;i<sizeX-2 ;i+=1 )
      for(int j=1;j<sizeZ-2;j+=1)
	height[i][j]=(height[i+1][j]+height[i][j+1]+height[i-1][j]+height[i][j-1])/4;
  
  //Berechnung von normalen Vektoren
  for(int i=1;i<sizeX-2;i+=1)
    for(int j=1;j<sizeZ-2 ;j+=1)
      {
	Vector v1 = Vector (widthX*(1),      height[i][j],      widthZ*(j) );
	Vector v2 = Vector (widthX*(i-1),    height[i-1][j],    widthZ*(j));
	Vector v3 = Vector (widthX*(i),      height[i][j+1],    widthZ*(j+1));
	Vector v4 = Vector (widthX*(i+1),    height[i+1][j],    widthZ*(j));
	Vector v5 = Vector (widthX*(i),      height[i][j-1],    widthZ*(j-1));
	
	Vector c1 = v2 - v1;
	Vector c2 = v3 - v1;
	Vector c3=  v4 - v1;
	Vector c4 = v5 - v1;
	Vector zero = Vector (0,0,0);
	normal_vectors[i][j]=c1.cross(v3-v5)+c2.cross(v4-v2)+c3.cross(v5-v3)+c4.cross(v2-v4);
	normal_vectors[i][j].normalize();
      }

  int snowheight=3;
  for ( int i = 0; i<sizeX; i+=1)
    for (int j = 0; j<sizeZ;j+=1)
      {	  
	Vector v1 = Vector (widthX*(i),      height[i][j]-20,       widthZ*(j)  -width/2);
	Vector v2 = Vector (widthX*(i+1),    height[i+1][j]-20,     widthZ*(j)  -width/2);
	Vector v3 = Vector (widthX*(i+1),    height[i+1][j+1]-20,   widthZ*(j+1)-width/2);
	Vector v4 = Vector (widthX*(i),      height[i][j+1]-20,     widthZ*(j+1)-width/2);
	float a[3];
	if(height[i][j]<snowheight){
	  a[0]=0;
	  a[1]=1.0-height[i][j]/10-.3;
	  a[2]=0;
	  glMaterialfv(GL_FRONT,GL_DIFFUSE,a);
	}
	else{
	    a[0]=1.0;
	    a[1]=1.0;
	    a[2]=1.0;
	    glMaterialfv(GL_FRONT,GL_DIFFUSE,a);
	    
	}
	glNormal3f(normal_vectors[i][j].x, normal_vectors[i][j].y, normal_vectors[i][j].z);
	glVertex3f(v1.x, v1.y, v1.z);
	if(height[i+1][j]<snowheight){
	  a[0]=0;
	  a[1] =1.0-height[i+1][j]/10-.3;
	  a[2]=0;
	  glMaterialfv(GL_FRONT,GL_DIFFUSE,a);
	}
	else{
	  a[0]=1.0;
	  a[1]=1.0;
	  a[2]=1.0;
	  glMaterialfv(GL_FRONT,GL_DIFFUSE,a);
	  
	}
	glNormal3f(normal_vectors[i+1][j].x, normal_vectors[i+1][j].y, normal_vectors[i+1][j].z);
	glVertex3f(v2.x, v2.y, v2.z);
	if(height[i+1][j+1]<snowheight){
	  a[0]=0;
	  a[1] =1.0-height[i+1][j+1]/10-.3;
	  a[2]=0;
	  glMaterialfv(GL_FRONT,GL_DIFFUSE,a);
	}
	else{
	  a[0]=1.0;
	  a[1]=1.0;
	  a[2]=1.0;
	  glMaterialfv(GL_FRONT,GL_DIFFUSE,a);
	  
	  
	}
	glNormal3f(normal_vectors[i+1][j+1].x, normal_vectors[i+1][j+1].y, normal_vectors[i+1][j+1].z);
	glVertex3f(v3.x, v3.y, v3.z);
	if(height[i][j+1]<snowheight){
	  a[0]=0;
	  a[1] =1.0-height[i+1][j+1]/10-.3;
	  a[2]=0;
	  glMaterialfv(GL_FRONT,GL_DIFFUSE,a);
	}
	else{
	  a[0]=1.0;
	  a[1]=1.0;
	  a[2]=1.0;
	  glMaterialfv(GL_FRONT,GL_DIFFUSE,a);
	}
	glNormal3f(normal_vectors[i][j+1].x, normal_vectors[i][j+1].y, normal_vectors[i][j+1].z);
	glVertex3f(v4.x, v4.y, v4.z);
	
      }
  glEnd();
  /*  
  glBegin(GL_LINES);
  for( float x = -128.0; x < 128.0; x += 25.0)
    {
      for( float y = -128.0; y < 128.0; y += 25.0)
	{
	  glColor3f(1,0,0);
	  glVertex3f(x,y,-128.0);
	  glVertex3f(x,y,0.0);
	  glColor3f(0.5,0,0);
	  glVertex3f(x,y,0.0);
	  glVertex3f(x,y,128.0);
	}
    }
  for( float y = -128.0; y < 128.0; y += 25.0)
    {
      for( float z = -128.0; z < 128.0; z += 25.0)
	{
	  glColor3f(0,1,0);
	  glVertex3f(-128.0,y,z);
	  glVertex3f(0.0,y,z);
	  glColor3f(0,0.5,0);
	  glVertex3f(0.0,y,z);
	  glVertex3f(128.0,y,z);
	}
    }
  for( float x = -128.0; x < 128.0; x += 25.0)
    {
      for( float z = -128.0; z < 128.0; z += 25.0)
	{
	  glColor3f(0,0,1);
	  glVertex3f(x,-128.0,z);
	  glVertex3f(x,0.0,z);
	  glColor3f(0,0,0.5);
	  glVertex3f(x,0.0,z);
	  glVertex3f(x,128.0,z);
	}
      
    }
  */  
  /*
  glBegin(GL_LINE_STRIP);
  glColor3f(1.0, 5.0, 1.0);
  for( int i = 0; i <= 30; i++)
    {
      glEvalCoord1f ((GLfloat) i/30.0);
    }
  glEnd();
  */

  trackManager->drawGraph(.01);
  trackManager->debug(2);
  /*  
  glBegin(GL_LINES);
  float i;
  for(i = 0.0; i<1; i+=.01)
    {
      printf("%f, %f, %f\n",tmpCurve->calcPos(i).x, tmpCurve->calcPos(i).y, tmpCurve->calcPos(i).z);
      glVertex3f(tmpCurve->calcPos(i).x, tmpCurve->calcPos(i).y, tmpCurve->calcPos(i).z);
    }
  glEnd();
  */
  glEndList();
  // LIGHT initialisation
  light = Light::getInstance();
  light->addLight(0);
  light->setAttenuation(QUADRATIC, 1.0);
  light->setAttenuation(CONSTANT, 2.0);
  light->setAttenuation(QUADRATIC, 1.0);
  light->setPosition(10.0, 10.0, 50.0);
  light->setDiffuseColor(1,1,1);
  //  light->addLight(1);
  //  light->setPosition(20, 10, -20);
  //  light->setDiffuseColor(0,0,0);
  light->debug();


}

/**
   \brief initializes a new World
*/
ErrorMessage World::init()
{
  this->bPause = false;
  CommandNode* cn = Orxonox::getInstance()->getLocalInput();
  cn->addToWorld(this);
  cn->enable(true);

  //glMap1f (GL_MAP1_VERTEX_3, 0.0, 1.0, 3, 4, &ctrlpoints[0][0]);
  //glEnable (GL_MAP1_VERTEX_3);
  
  //theNurb = gluNewNurbsRenderer ();
  //gluNurbsProperty (theNurb, GLU_NURBS_MODE, GLU_NURBS_TESSELLATOR);
  //gluNurbsProperty (theNurb, GLU_NURBS_VERTEX, vertexCallback );

}


/**
   \brief starts the World
*/
ErrorMessage World::start()
{
  printf("World::start() - starting current World: nr %i\n", this->debugWorldNr);
  this->bQuitOrxonox = false;
  this->bQuitCurrentGame = false;
  this->mainLoop();
}

/**
   \brief stops the world.

   This happens, when the player decides to end the Level.
*/
ErrorMessage World::stop()
{
  printf("World::stop() - got stop signal\n");
  this->bQuitCurrentGame = true;
}

/**
   \brief pauses the Game
*/
ErrorMessage World::pause()
{
  this->isPaused = true;
}

/**
   \brief ends the pause Phase
*/
ErrorMessage World::resume()
{
  this->isPaused = false;
}

/**
   \brief destroys the World
*/
ErrorMessage World::destroy()
{
  delete trackManager;
}

/**
   \brief shows the loading screen
*/
void World::displayLoadScreen ()
{
  printf ("World::displayLoadScreen - start\n"); 
  
  //GLMenuImageScreen* 
  this->glmis = GLMenuImageScreen::getInstance();
  this->glmis->init();
  this->glmis->setMaximum(10);
  this->glmis->draw();
 
  printf ("World::displayLoadScreen - end\n"); 
}

/**
   \brief removes the loadscreen, and changes over to the game

   \todo take out the delay
*/
void World::releaseLoadScreen ()
{
  printf ("World::releaseLoadScreen - start\n"); 
  this->glmis->setValue(this->glmis->getMaximum());
  SDL_Delay(500);
  printf ("World::releaseLoadScreen - end\n"); 
}


/** 
    \brief checks for collisions
    
    This method runs through all WorldEntities known to the world and checks for collisions 
    between them. In case of collisions the collide() method of the corresponding entities 
    is called.
*/
void World::collide ()
{
  /*
  List *a, *b;
  WorldEntity *aobj, *bobj;
   
  a = entities;
  
  while( a != NULL)
    {
      aobj = a->nextElement();
      if( aobj->bCollide && aobj->collisioncluster != NULL)
	{
	  b = a->nextElement();
	  while( b != NULL )
	    {
	      bobj = b->nextElement();
	      if( bobj->bCollide && bobj->collisioncluster != NULL )
		{
		  unsigned long ahitflg, bhitflg;
		  if( check_collision ( &aobj->place, aobj->collisioncluster, 
					&ahitflg, &bobj->place, bobj->collisioncluster, 
					&bhitflg) );
		  {
		    aobj->collide (bobj, ahitflg, bhitflg);
		    bobj->collide (aobj, bhitflg, ahitflg);
		  }
		}
	      b = b->nextElement();
	    }
	}
      a = a->enumerate();
    }
  */
}

/** 
    \brief runs through all entities calling their draw() methods
*/
void World::draw ()
{
  /* draw entities */
  WorldEntity* entity;
  entity = this->entities->enumerate();
  while( entity != NULL ) 
    { 
      if( entity->bDraw ) entity->draw();
      entity = this->entities->nextElement();
    } 
  
  glCallList (objectList);
  //! \todo skysphere is a WorldEntity and should be inside of the world-entity-list.
  skySphere->draw();

  testFont->printText(0, 0, 1, "orxonox_" PACKAGE_VERSION);

}


/**
   \brief function to put your own debug stuff into it. it can display informations about
   the current class/procedure
*/
void World::debug()
{
  printf ("World::debug() - starting debug\n");
  PNode* p1 = NullParent::getInstance ();
  PNode* p2 = new PNode (new Vector(2, 2, 2), p1);
  PNode* p3 = new PNode (new Vector(4, 4, 4), p1);
  PNode* p4 = new PNode (new Vector(6, 6, 6), p2);

  p1->debug ();
  p2->debug ();
  p3->debug ();
  p4->debug ();

  p1->shiftCoor (new Vector(-1, -1, -1));

  printf("World::debug() - shift\n");
  p1->debug ();
  p2->debug ();
  p3->debug ();
  p4->debug ();
  
  p1->update (1);

  printf ("World::debug() - update\n");
  p1->debug ();
  p2->debug ();
  p3->debug ();
  p4->debug ();

  p2->shiftCoor (new Vector(-1, -1, -1));
  p1->update (2);

  p1->debug ();
  p2->debug ();
  p3->debug ();
  p4->debug ();

  p2->setAbsCoor (new Vector(1,2,3));


 p1->update (2);

  p1->debug ();
  p2->debug ();
  p3->debug ();
  p4->debug ();

  p1->destroy ();
  
  
  /*
  WorldEntity* entity;
  printf("counting all entities\n");
  printf("World::debug() - enumerate()\n");
  entity = entities->enumerate();  
  while( entity != NULL ) 
    { 
      if( entity->bDraw ) printf("got an entity\n");
      entity = entities->nextElement();
    }
  */
}


/**
  \brief main loop of the world: executing all world relevant function

  in this loop we synchronize (if networked), handle input events, give the heart-beat to
  all other member-entities of the world (tick to player, enemies etc.), checking for
  collisions drawing everything to the screen.
*/
void World::mainLoop()
{
  this->lastFrame = SDL_GetTicks ();
  printf("World::mainLoop() - Entering main loop\n");
  while( !this->bQuitOrxonox && !this->bQuitCurrentGame) /* \todo implement pause */
    {
      // Network
      this->synchronize ();
      // Process input
      this->handleInput ();
      if( this->bQuitCurrentGame || this->bQuitOrxonox)
	{
	  printf("World::mainLoop() - leaving loop earlier...\n");
	  break;
	}
      // Process time
      this->timeSlice ();
      // Process collision
      this->collide ();
      // Draw
      this->display ();
  
      for( int i = 0; i < 5000000; i++) {}
      /* \todo this is to slow down the program for openGl Software emulator computers, reimplement*/
    }
  printf("World::mainLoop() - Exiting the main loop\n");
}


/**
   \brief synchronize local data with remote data
*/
void World::synchronize ()
{
  // Get remote input
  // Update synchronizables
}


/**
   \brief run all input processing

   the command node is the central input event dispatcher. the node uses the even-queue from
   sdl and has its own event-passing-queue.
*/
void World::handleInput ()
{
  // localinput
  CommandNode* cn = Orxonox::getInstance()->getLocalInput();
  cn->process();
  // remoteinput
}


/**
   \brief advance the timeline

   this calculates the time used to process one frame (with all input handling, drawing, etc)
   the time is mesured in ms and passed to all world-entities and other classes that need
   a heart-beat.
*/
void World::timeSlice ()
{
  Uint32 currentFrame = SDL_GetTicks();
  if(!this->bPause)
    {
      Uint32 dt = currentFrame - this->lastFrame;
      
      if(dt > 0)
	{
	  float fps = 1000/dt;
	  printf("fps = %f\n", fps);
	}
      else
	{
	  /* the frame-rate is limited to 100 frames per second, all other things are for
	     nothing.
	  */
	  printf("fps = 1000 - frame rate is adjusted\n");
	  SDL_Delay(10);
	  dt = 10;
	}
      //this->timeSlice (dt);
      
      /* function to let all entities tick (iterate through list) */
      WorldEntity* entity;
      float seconds = dt / 1000.0;      
      this->nullParent->update (seconds);
      entity = entities->enumerate(); 
      while( entity != NULL) 
	{ 
	  entity->tick (seconds);
	  entity = entities->nextElement();
	}
      skySphere->updatePosition(localCamera->absCoordinate);
      
      /* update tick the rest */
      this->localCamera->timeSlice(dt);
      this->trackManager->tick(dt);
    }
  this->lastFrame = currentFrame;
}


/**
   \brief render the current frame
   
   clear all buffers and draw the world
*/
void World::display ()
{
  // clear buffer
  glClear( GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);
  // set camera
  this->localCamera->apply ();
  // draw world
  this->draw();
  // draw HUD
  /* \todo draw HUD */
  // flip buffers
  SDL_GL_SwapBuffers();
  //SDL_Surface* screen = Orxonox::getInstance()->getScreen ();
  //SDL_Flip (screen);
}


/**
   \brief add and spawn a new entity to this world
   \param entity to be added
*/
void World::spawn(WorldEntity* entity)
{
  if( this->nullParent != NULL && entity->parent == NULL)
    this->nullParent->addChild (entity);

  this->entities->add (entity);

  entity->postSpawn ();
}


/**
   \brief add and spawn a new entity to this world
   \param entity to be added
   \param absCoor At what coordinates to add this entity.
   \param absDir In which direction should it look.
*/
void World::spawn(WorldEntity* entity, Vector* absCoor, Quaternion* absDir)
{
  entity->setAbsCoor (absCoor);
  entity->setAbsDir (absDir);
  
  if( this->nullParent != NULL && entity->parent == NULL)
    this->nullParent->addChild (entity);

  this->entities->add (entity);

  entity->postSpawn ();
}



/**
  \brief commands that the world must catch
  \returns false if not used by the world
*/
bool World::command(Command* cmd)
{
  return false;
}

void World::setPath( char* name)
{
	path = name;
}

char* World::getPath()
{
	return path;
}
