

/* 
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Patrick Boenzli
   co-programmer: Christian Meyer
*/

#include <iostream>

#include "world_entity.h"
#include "stdincl.h"
//#include "collision.h"

using namespace std;

/**
   \brief standard constructor
   
   Every derived contructor HAS to call the previous one supplying the isFree parameter. This is necessary to distunguish
   between free and bound entities. The difference between them is simply the fact that the movement of a free entity is
   not bound to the track of a world. Use this to implement projectile or effect classes that do not have to travel along the track.
   To specify an entity to be free or bound set the default parameter in the declaration of the constructor.
   Theoretically you should never have to call the constructor of an Entity directly, for it is called by the spawn() function of the World
   class. So if you want to create a new entity at any time, call World::spawn(). It will handle everything that is necessary.
*/
WorldEntity::WorldEntity (bool isFree) : bFree(isFree)
{
  this->setClassName ("WorldEntity");
  this->bDraw = true;
  //  collisioncluster = NULL;
}

/**
   \brief standard destructor
*/
WorldEntity::~WorldEntity ()
{
  // if( collisioncluster != NULL) delete collisioncluster;
}

/**
   \brief get the Location of the WorldEntity
   \return a pointer to location
*/
/*PN
Location* WorldEntity::getLocation ()
{
  return &loc;
}
*/

/**
   \brief get the Placement of the WorldEntity
   \return a pointer to placement
*/
 /*PN
Placement* WorldEntity::getPlacement ()
{
  return &place;
}
 */
/**
   \brief query whether the WorldEntity in question is free
   \return true if the WorldEntity is free or false if it isn't
*/
bool WorldEntity::isFree ()
{
  return bFree;
}

/**
   \brief set the WorldEntity's collision hull
   \param newhull: a pointer to a completely assembled CollisionCluster
   
   Any previously assigned collision hull will be deleted on reassignment
*/
/*
void WorldEntity::setCollision (CollisionCluster* newhull)
{
  if( newhull == NULL) return;
  if( collisioncluster != NULL) delete collisioncluster;
  collisioncluster = newhull;
}
*/

/**
   \brief this method is called every frame
   \param time: the time in seconds that has passed since the last tick
   
   Handle all stuff that should update with time inside this method (movement, animation, etc.)
*/
void WorldEntity::tick(float time) 
{
}


/** 
    \brief process draw function 
*/
void WorldEntity::processDraw ()
{
  this->draw ();
  PNode* pn = this->children->enumerate ();
  while( pn != NULL) 
    { 
      ((WorldEntity*)pn)->processDraw ();
      pn = this->children->nextElement();
    } 
}

/**
   \brief sets the drawable state of this entity.
   \param bDraw TRUE if draweable, FALSE otherwise
*/
void WorldEntity::setDrawable (bool bDraw)
{
  this->bDraw = bDraw;
}


/**
   \brief the entity is drawn onto the screen with this function
   
   This is a central function of an entity: call it to let the entity painted to the screen. Just override this function with whatever you want to be drawn.
*/
void WorldEntity::draw() 
{}

/**
   \brief this function is called, when two entities collide
   \param other: the world entity with whom it collides
   \param ownhitflags: flags to the CollisionCluster subsections that registered an impact
   \param otherhitflags: flags to the CollisionCluster subsections of the other entity that registered an impact

   Implement behaviour like damage application or other miscellaneous collision stuff in this function
*/
void WorldEntity::collide(WorldEntity* other, Uint32 ownhitflags, Uint32 otherhitflags) {}

/**
   \brief this function is called, when the ship is hit by a waepon
   \param weapon: the laser/rocket/shoot that hits.
   \param loc: place where it is hit

   calculate the damage depending
*/
void WorldEntity::hit(WorldEntity* weapon, Vector loc) {}

/**
   \brief this function is called when the entity is to be destroied
   
   This can be called, if eg. something realy bad happens :)
*/
void WorldEntity::destroy() {}


/**
   \brief this is called immediately after the Entity has been constructed and initialized
   
   Put any initialisation code that requires knowledge of location (placement if the Entity is free) and owner of the entity here.
   DO NOT place such code in the constructor, those variables are set AFTER the entity is constucted.
*/
void WorldEntity::postSpawn ()
{
}

/**
   \brief this handles incoming command messages
   \param cmd: a pointer to the incoming Command structure
   
   Put all code that handles Command messages here, this will mainly be called by the assigned CommandNode but can also be used
   to send commands from one WorldEntity to another.
*/
void WorldEntity::command (Command* cmd)
{
}

/**
   \brief this is called by the local Camera to determine the point it should look at on the WorldEntity
   \param locbuf: a pointer to the buffer to fill with a location to look at
	
   You may put any Location you want into locbuf, the Camera will determine via the corresponding Track how
   to look at the location you return with this.
*/
/*PN
void WorldEntity::getLookat (Location* locbuf)
{
}
*/

/**
   \brief this method is called by the world if the WorldEntity leaves valid gamespace
   
   For free entities this means it left the Track boundaries. With bound entities it means its Location adresses a
   place that is not in the world anymore. In both cases you might have to take extreme measures (a.k.a. call destroy).
*/
void WorldEntity::leftWorld ()
{
}
