/*! 
    \file light.h
    \brief Handles Lights.

    A Light is one of the more important things in a 3D-environment,
    without it one sees nothing :)
    It is here for diffuse-, specular- and Bump-Mappings.
*/

#ifndef _LIGHT_H
#define _LIGHT_H

#include "world_entity.h"
#include "glincl.h"

//! The maximum number of Lights this OpenGL-implementation supports
#define NUMBEROFLIGHTS GL_MAX_LIGHTS

//! Enumerator for the attenuation-Type.
/**
   CONSTANT means GL_CONSTANT_ATTENUATION
   LINEAR means   GL_LINEAR_ATTENUATION
   QUADRATIC means GL_QUADRATIC_ATTENUATION
*/
enum AttenuationType {CONSTANT, LINEAR, QUADRATIC};

// FORWARD DEFINITIONS //
class Vector;

//! A class that handles Lights
/**
   A Light is a source that emits light rays (photons)
*/
class Light : public WorldEntity
{
 private:
  //! A struct that holds information about a Light
  struct LightValue
  {
    int lightNumber;            //!< The number of this Light.
    GLfloat lightPosition[4];   //!< The Position of this Light.
    GLfloat diffuseColor[4];    //!< The Diffuse Color this Light emmits.
    GLfloat specularColor[4];   //!< The specular Color of this Light.
    AttenuationType attenuationType;//!< The AttenuationType of this Light.
    float attenuationFactor;    //!< The Factor the attenuation should have.
    GLfloat spotDirection[4];   //!< The direction of the Spot Light.
    GLfloat spotCutoff;         //!< The cutoff Angle of the Light Source
  };

  static Light* singletonRef;    //!< This is the LightHandlers Reference.
  GLfloat ambientColor[4];       //!< The ambient Color of the scene.


  Light(void);

  void init(int LightNumber);  
  LightValue** lights;           //!< An array of Lenght NUMBEROFLIGHTS, that holds pointers to all LightValues.
  LightValue* currentLight;      //!< The current Light, we are working with.
  
 public:
  static Light* getInstance();
  ~Light(void);

  // set Attributes
  int addLight(void);
  int addLight(int lightNumber);
  void editLightNumber(int lightNumber);
  void deleteLight(void);
  void deleteLight(int lightNumber);

  void setPosition(Vector position);
  void setPosition(GLfloat x, GLfloat y, GLfloat z);
  void setDiffuseColor(GLfloat r, GLfloat g, GLfloat b);
  void setSpecularColor(GLfloat r, GLfloat g, GLfloat b);
  void setAttenuation(AttenuationType type, float factor);
  void setAmbientColor(GLfloat r, GLfloat g, GLfloat b);
  void setSpotDirection(Vector direction);
  void setSpotCutoff(GLfloat cutoff);

  // get Attributes
  Vector getPosition(void);
  /**
     \returns the Position of Light 
     \param lightNumber lightnumber
  */
  inline Vector getPosition(int lightNumber)
    {
      if (!this->lights[lightNumber])
	return Vector(.0,.0,.0);
      return Vector(this->lights[lightNumber]->lightPosition[0],
		    this->lights[lightNumber]->lightPosition[1],
		    this->lights[lightNumber]->lightPosition[2]);
    }

  void debug(void);
};

#endif /* _LIGHT_H */
