/*!
 \file orxonox_gui_gtk.h
 \brief Contains all th different Widgets.
*/
#ifndef _ORXONOX_GUI_GTK_H
#define _ORXONOX_GUI_GTK_H

#if HAVE_CONFIG_H 
#include <config.h> 
#endif

//! verbose level, soon obsolete
extern int verbose; // soon obsolete here

#include "debug.h"

#ifdef HAVE_GTK2
#include <gtk/gtkmain.h>
#include <gtk/gtkwindow.h>
#include <gtk/gtkframe.h>
#include <gtk/gtkhbox.h>
#include <gtk/gtkvbox.h>
#include <gtk/gtkbutton.h>
#include <gtk/gtkcheckbutton.h>
#include <gtk/gtkhscale.h>
#include <gtk/gtkoptionmenu.h>
#include <gtk/gtkmenu.h>
#include <gtk/gtkmenuitem.h>
#include <gtk/gtklabel.h>
#include <gtk/gtkimage.h>
#include <gtk/gtkeventbox.h>
#include <gtk/gtkprogressbar.h>
#endif /* HAVE_GTK2 */

bool initGUI(int argc, char* argv[]);
bool mainloopGUI(void);

//! This is the topmost object that can be displayed all others are derived from it.
class Widget
{
 private:

 public:
  virtual ~Widget(void);
  void init(void);
  void destroy(void);

  void show(void);
  void hide(void);
  void setSize(int width, int height);
  virtual void setTitle(char* title) = 0;  //!< An abstract Function, that sets the title of Widgets.

  Widget* findWidgetByName(char* name, unsigned int depth);
  void walkThrough(void(*function)(Widget*), unsigned int depth);
  void walkThrough(void(*function)(Widget*, void*), void* data, unsigned int depth);
  static void listOptionsAndGroups(Widget* widget);
  static void listOptions(Widget* widget);
  static void listOptions(Widget* widget, void* data);
  Widget* findOptionByNumber(int* number, unsigned int depth);
  static void listGroups(Widget* widget);
  static void listGroups(Widget* widget, void* data);
  Widget* findGroupByNumber(int* number, unsigned int depth);
  static void setOptions(Widget* widget);
  static void flagCheck(Widget* widget, void* flagName);
  
#ifdef HAVE_GTK2
  // Connection - Functions

  gulong connectSignal(char* event, gint(*signal)(GtkWidget*, GdkEvent*, void* ));
  gulong connectSignal(char* event, gint(*signal)(GtkWidget*, Widget* ));
  gulong connectSignal(char* event, void* extObj, gint(*signal)(GtkWidget*, GdkEvent*, void* ));
  gulong connectSignal(char* event, void* extObj, gint(*signal)(GtkWidget*, void* ));
  gulong connectSignal(char* event, void* extObj, gint(*signal)(GtkWidget*, GdkEventKey*, void* ));
  void disconnectSignal(gulong signalID);

  // Signals
  static gint doNothingSignal(GtkWidget* widget, GdkEvent* event, void* nothing);
#endif /* HAVE_GTK2 */

  Widget* next;                     //!< next always points to the next Widget in the list. Every Widget has a next one, or has NULL as next
#ifdef HAVE_GTK2
  GtkWidget* widget;                //!< widget is the gtk_widget that the specific Object Contains.
#endif /* HAVE_GTK2 */
  int isOption;                     //!< with this Paramenter one can set the option-type: -2:Container, -1: Box, 0: not an Option, 1: Bool-option, 2: int-option, 3: float option, 4:char option, 5: char* option

  //! defines isOption states
  enum option { containerType = -2, boxType = -1, nothingType = 0, boolType = 1, intType = 2}; 
  char* title;                      //!< The name of the Widget. Some do display it, Options need it to save;
};

//! This is a Packer Object, which has the ability to Pack other Widgets into itself.
class Packer : public Widget
{
 public:
  Widget* down; //!< this points to the Widget below this.
  char* groupName; //!< For each Packer you can specify a Groupname under which the lowerWidgets will be saved.

  void init(void);
  void destroy(void);

  void setGroupName(char* name);
  char* getGroupName(void);

  virtual void fill(Widget* lowerWidget) = 0; //!< An abstract function, that fills Packers.
};

//! This is a Container Class, it can contain one sub-Widget: down.
/**
 * A Container is a Widget that can hold a subWidget in addition to a next-Widget.
 * The Container can by itself not be displayed created or used.
 * The derived classes of Container can be displayed
*/
class Container : public Packer
{
 private:
  int borderwidth;          //!< The width of The Container Boarder.
  int policy;               //!< The Update Policy of a Container.
  
 public:
  void init(void);
  void destroy(void);

  //  void setBorderWidth(int borderwidth);
  //  virtual void setTitle(char* title) = 0;
  void fill(Widget* lowerWidget);
};

//! Window is the class that creates new Windows, and handels them
/**
 * A Window is a class derived from Container that contains a window-widget.
 * It has the ability to hold one sub-object
 */
class Window : public Container
{
 private:
  bool isOpen;                      //!< A bool Variable that checks if a Window is already open.
 public:
  static Window* mainWindow;        //!< Pointer to the First Window that was opened. By default this should be the GUI's main-Window.
  static void addWindow(Window* windowToAdd);

  Window(void);
  Window(char* windowName);
  ~Window(void);
  void init(void);
  void destroy(void);
  
  void setTitle(char* title);
  void showall(void);
  void open(void);
  void close(void);

#ifdef HAVE_GTK2
  // Signals
  static gint windowOpen(GtkWidget* widget, GdkEvent* event, void* window);
  static gint windowClose(GtkWidget* widget, GdkEvent* event, void* window);
#endif /* HAVE_GTK2 */
};

//! Frame is the class that handles frames
/**
 * A Frame is an object, that has a border and if you like a name on it.
 * It can contain a Widget, which means that you can insert anything you like inside of a frame
 */
class Frame :public Container
{
 public:
  Frame(void);
  Frame(char* frameName);
  ~Frame(void);
  void init(void);
  void destroy(void);
  
  void setTitle(char* title);
};

//! EventBox is a Container that can Handle all Events happening inside of it.
/**
 * Example: if you have a picture, and you want it to catch mouse-clicks, you have to pack it inside a EventBox
 */
class EventBox : public Container
{
 public:
  EventBox(void);
  EventBox(char* eventBoxName);
  ~EventBox(void);
  void init(void);
  void destroy(void);
  
  void setTitle(char* title);
};

//! A Box can contain multiple Widgets
/**
   A Box can Contain multiple Widgets, that are ordered either horizontally or vertically
   I defined the standartbox to be horizontally.
   A Box is always filled left->right(horizontally) or up->down(vertically)
*/
class Box : public Packer
{
 public:
  Box(void);
  Box(char boxtype);
  ~Box(void);
  void init(char boxtype);
  void destroy(void);
  
  void fill(Widget* lowerWidget);

  void setTitle(char* title);
};

//! An Option is a Widget that contains something that may change its state.
/**
 * Options are the essence of a GUI, they: Configure, Start, Quit, Execute, and make it worth something
 */
class Option : public Widget
{
 protected:
  bool saveable;  //!< Options can be Saved.
  
 public:
  void init(void);
  void destroy(void);

  int value; //!< every option has a value either true or false(0,1) or something else like 25 for 25% of the volume
  char* flagName; //!< options have a flag name that will be appendet if you start the Program from the GUI.
  char* flagNameShort; //!< like flag_name but shorter
  int defaultValue; //!< A default value is good, for hiding a option if it is not needed.(hidden if value == default_value)

  void saveability(void);
  void saveability(bool isSaveable);
  bool isSaveable(void);
  void setFlagName(char* flagname, int defaultvalue);
  void setFlagName(char* flagname, char* flagnameshort, int defaultvalue);
  virtual void redraw(void) = 0; //!< A Option must be able to redraw itself.
  virtual void changeOption(void) = 0; //!< What to do, if an Option is Changed. eacht option decides for itself.
#ifdef HAVE_GTK2
    // Signals  
  static gint OptionChange(GtkWidget* widget, Widget* option); //!< Signal for Options that change.
#endif /* HAVE_GTK2 */
};

//! Buttons can be pressed, and released.
/**
 * Buttons are mainly there for executing some action like Starting the Programm, or Quiting it.
 */
class Button : public Option 
{
 public:
  Button(char* buttonname);
  ~Button(void);
  void init(void);
  void destroy(void);

  void setTitle(char* title);
  void redraw(void);
  void changeOption(void);
};

//! CheckButtons are a key in configuring bool Variables
/** CheckButtons can configure bool Variables like wireframe on/off, enable_sound etc.
 */
class CheckButton : public Option
{
 public:
  CheckButton(char* buttonname);
  ~CheckButton(void);
  void init(void);
  void destroy(void);

  void setTitle(char* title);
  bool isActive(void);           //!< a Bool value to see, if this CheckButton is active.
  void redraw(void);
  void changeOption(void);
};

//! Sliders are Options that can be modified in their value
/**
 * good for volume, brightness, etc.
 */
class Slider : public Option
{
 private:
  int start;                            //!< The beginning of the Slider-range.
  int end;                              //!< The end of the Slider-range.
 public:
  Slider(char* slidername, int start, int end);
  ~Slider(void);
  void init(int start, int end);
  void destroy(void);

  void setTitle(char* title);
  void setValue(int value);
  void redraw(void);
  void changeOption(void);
};

//! A Menu is an Option that has a dropdown menu, where you can chose between different Items
class Menu : public Option
{
 private:
#ifdef HAVE_GTK2
  GtkWidget* menu;                      //!< The menu That will hold the Options.
  GtkWidget* item;                      //!< One Item From a Menu.
#endif /* HAVE_GTK2 */
  va_list itemlist;                     //!< The list to readin multiple Options.
  
 public:
  Menu(char* menuname, ...);
  ~Menu(void);
  void init(void);
  void destroy(void);
  
  void setTitle(char* title);
  void addItem(char* itemName);
  void redraw(void);
  void changeOption(void);
};

//! A OptionLabel is a simple Label, that holds a char*, and will be updated, if changed.
class OptionLabel : public Option
{
 private:

 public:
  OptionLabel(char* label, char* value);
  ~OptionLabel(void);
  void init(void);
  void destroy(void);
  
  
  void setValue(char* newValue);
  void setTitle(char* title);
  void redraw(void);
  void changeOption(void);

  char* cValue;                          //!< The Value the Label will have. \todo make private

};

//! A label is a Widget, that displays a text
class Label : public Widget
{
 public:
  Label(void);
  Label(char* text);
  ~Label(void);
  void init(void);
  void destroy(void);
  
  void setTitle(char* text);
  void ereaseText(void);
  void appendText(char* textToAppend);
  void appendInt(int intToAppend);
  char* getText(void);
};

//! A ProgressBar is a Widget, that can display a Progress
class ProgressBar : public Widget
{
 public:
  ProgressBar(void);
  ProgressBar(char* label);
  ~ProgressBar(void);
  void init(void);
  void destroy(void);

  void setTitle(char* title);
  void setProgress(double progress);
  void setTotalSize(double totalSize);
  double getProgress(void);

 private:
  double totalSize;         //!< The total Size of a download Bar
  double progress;          //!< The progress of a Bar.
#ifdef HAVE_GTK2
  GtkAdjustment* adjustment;
#endif /* HAVE_GTK2 */
};

//! Image is the keeper of one Image 
/** 
 * Images are mighty cool.
 * Images can help you lighten up the Programming process, and will give everyone a better impression of the Software.
 */
class Image : public Widget
{
 public:
  Image(char* imgaename);
  ~Image(void);
  void init(void);
  void destroy(void);

  void setTitle(char* title);
};

//gint orxonox_gui_quit(GtkWidget* widget, GdkEvent* event, gpointer data);

#endif /* _ORXONOX_GUI_GTK_H */
