/*!
  \file model.h
  \brief Contains the Model Class that handles 3D-Models
*/

#ifndef _MODEL_H
#define _MODEL_H

#include "material.h"

// FORWARD DEFINITION //
class Array;
class Vector;

using namespace std;

//! Specification of different primitives the Model knows
enum PRIMITIVE {PLANE, CUBE, SPHERE, CYLINDER};

//! Class that handles 3D-Models. it can also read them in and display them.
class Model
{
 public:
  Model(void);
  Model(PRIMITIVE type);
  Model(char* modelName);
  virtual ~Model(void);

  void setName(const char* name);
  
  void draw(void) const;
  void draw(int groupNumber) const;
  void draw(char* groupName) const;
  int getGroupCount() const;

 protected:
  char* name;            //!< This is the name of the Model.
  bool finalized;        //!< Sets the Object to be finalized.
  
  //! This is the placeholder of one Vertex beloning to a Face.
  struct FaceElement
  {
    int vertexNumber;    //!< The number of the Vertex out of the Array* vertices, this vertex points to.
    int normalNumber;    //!< The number of the Normal out of the Array* normals, this vertex points to.
    int texCoordNumber;  //!< The number of the textureCoordinate out of the Array* vTexture, this vertex points to.
    FaceElement* next;   //!< Point to the next FaceElement in this List.
  };

  //! This is the placeholder of a Face belonging to a Group of Faces.
  struct Face
  {
    int vertexCount;        //!< The Count of vertices this Face has.
    FaceElement* firstElem; //!< Points to the first Vertex (FaceElement) of this Face.

    char* materialString;   //!< The Name of the Material to which to Change.

    Face* next;             //!< Pointer to the next Face.
  }; 

  //! Group to handle multiple Models per obj-file.
  struct Group
  {
    char* name;         //!< the Name of the Group. this is an identifier, that can be accessed via the draw (char* name) function.

    unsigned int listNumber;//!< The number of the GL-List this Group gets.
    Face* firstFace;    //!< The first Face in this group.
    Face* currentFace;  //!< The current Face in this Group (the one we are currently working with.)
    int faceMode;       //!< The Mode the Face is in: initially -1, 0 for FaceList opened, 1 for Material,  3 for triangle, 4 for Quad, 5+ for Poly \todo ENUM...
    int faceCount;      //!< The Number of Faces this Group holds.

    Group* next;        //!< Pointer to the next Group.
  };


  Array* vertices;      //!< The Array that handles the Vertices.
  int verticesCount;    //!< A global Counter for vertices.
  Array* normals;       //!< The Array that handles the Normals.
  Array* vTexture;      //!< The Array that handles the VertexTextureCoordinates.

  
  Group* firstGroup;    //!< The first of all groups.
  Group* currentGroup;  //!< The currentGroup. this is the one we will work with.
  int groupCount;       //!< The Count of Groups.

  Material* material;   //!< Initial pointer to the Material. This can hold many materials, because Material can be added with Material::addMaterial(..)
  float scaleFactor;    //!< The Factor with which the Model should be scaled. \todo maybe one wants to scale the Model after Initialisation

  bool initialize(void);
  bool initGroup(Group* group);
  bool initFace (Face* face);
  bool cleanup(void);
  bool cleanupGroup(Group* group);
  bool cleanupFace(Face* face);
  bool cleanupFaceElement(FaceElement* faceElem);

 public:
  bool addGroup(char* groupString);
  bool addVertex(char* vertexString);
  bool addVertex(const float x, const float y, const float z);
  bool addFace(char* faceString);
  bool addFace(const float faceElemCount, int type, ...);
  bool addVertexNormal(char* normalString);
  bool addVertexNormal(const float x, const float y, const float z);
  bool addVertexTexture(char* vTextureString);
  bool addVertexTexture(const float u, const float v);
  bool addUseMtl(char* mtlString);
  void finalize(void);

 protected:
  bool importToGL(void);
  bool addGLElement(FaceElement* elem);

  bool buildVertexNormals(void);

  void cubeModel(void);
  void sphereModel(void);
  void cylinderModel(void);
};

#endif
