/*! 
    \file collision.h
    \brief Basic collision detection
*/ 

#ifndef _COLLISION_H
#define _COLLISION_H

#include "vector.h"
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

//! Tree structure used by the CollisionCluster
typedef struct CCTree
{
  unsigned long n;
  union
  {
  struct CCTree** b;
  unsigned long ID;
  } data;
  float r;
  Vector m;
} CCTree;

//! Basic collision detection class
/**
  This class implements a more or less efficient collision system based on nested hitzones.
  Instead of using a brute force approach (try if any hitzone intersects with any other hitzone)
  this features a tree of spheric hitzones. Only some of them are actually the solid groundstones
  the collision model bases on, the others serve to group them into sets of spheres that are only
  checked for collision when the assigned top level sphere has registered a collision, preventing
  unnessessary checks (like between every sphere of two collision clusters at the other end of the world)
  from being performed.
  The CollisionCluster features collision detection between multiple CollisionClusters as well as
  traceing a collision between a line of defined length and a cluster. In both cases the base spheres
  that have intersected are marked with a flag in an unsigned long for hitlocation queries. In the case
  of a trace, the exact point of interception is returned as well.
*/
class CollisionCluster {
  
  CCTree* root;
  
  
 public:
  CollisionCluster (float r, Vector m); // simple cluster
  CollisionCluster (char* filename); // get cluster from file
  ~CollisionCluster ();
  
  int store (char* filename);
  
  friend bool ccTreeTrace( const Placement* p, CCTree* t, unsigned long* hitflags, const Line* trace, Vector* impactpoint);
  friend bool ccTreeIterate(const Placement* pa, CCTree* ta, unsigned long* ahitflags, const Placement* pb, CCTree* tb, unsigned long* bhitflags);
  friend bool checkTrace (const Placement* pa, const CollisionCluster* a, unsigned long* ahitflags, const Line* trace, Vector* impactpoint);
  friend bool checkCollision (const Placement* pa, const CollisionCluster* a, unsigned long* ahitflags, const Placement* pb, const CollisionCluster* b, unsigned long* bhitflags);
};

bool sphereSphereCollision( Vector m1, float r1, Vector m2, float r2);
bool traceSphereCollision( Vector m, float r, const Line* l, Vector* impactpoint);

void setflag( unsigned long* flags, unsigned long ID);

void freeCCTree( CCTree* tree);
CCTree* loadCCTree (FILE* stream);
int saveCCTree (CCTree* tree, FILE* stream);

#endif /* _COLLISION_H */
