/*!
  \file model.h
  \brief Contains the Model Class that handles 3D-Models
*/

#ifndef _MODEL_H
#define _MODEL_H

#include "../stdincl.h"

#include "array.h"
#include "material.h"
#include "vector.h"

using namespace std;



//! Class that handles 3D-Models. it can also read them in and display them.
class Model
{
 public:
  Model ();
  Model (char* fileName);
  Model(char* fileName, float scaling);
  ~Model ();
  
  void draw (void) const;
  void draw (int groupNumber) const;
  void draw (char* groupName) const;
  int getGroupCount() const;

 private:
  //! This is the placeholder of one Vertex beloning to a Face.
  struct FaceElement 
  {
    int vertexNumber;    //!< The number of the Vertex out of the Array* vertices, this vertex points to.
    int normalNumber;    //!< The number of the Normal out of the Array* normals, this vertex points to.
    int texCoordNumber;  //!< The number of the textureCoordinate out of the Array* vTexture, this vertex points to.
    FaceElement* next;   //!< Point to the next FaceElement in this List.
  };

  //! This is the placeholder of a Face belonging to a Group of Faces.
  /**
     \todo take Material to a call for itself.

     This can also be a Material-Change switch. 
     That means if you want to change a Material inside of a group,
     you can create an empty face and apply a material to it, and the Importer will cahnge Colors
  */
  struct Face 
  {
    int vertexCount;        //!< The Count of vertices this Face has.
    FaceElement* firstElem; //!< Points to the first Vertex (FaceElement) of this Face.

    char* materialString;   //!< The Name of the Material to which to Change.

    Face* next;             //!< Pointer to the next Face.
  }; 

  //! Group to handle multiple Models per obj-file.
  struct Group
  {
    char* name;         //!< the Name of the Group. this is an identifier, that can be accessed via the draw (char* name) function.

    GLuint listNumber;  //!< The number of the GL-List this Group gets.
    Face* firstFace;    //!< The first Face in this group.
    Face* currentFace;  //!< The current Face in this Group (the one we are currently working with.)
    int faceMode;       //!< The Mode the Face is in: initially -1, 0 for FaceList opened, 1 for Material,  3 for triangle, 4 for Quad, 5+ for Poly \todo ENUM...
    int faceCount;      //!< The Number of Faces this Group holds.

    Group* next;        //!< Pointer to the next Group.
  };


  Array* vertices;      //!< The Array that handles the Vertices.
  int verticesCount;    //!< A global Counter for vertices.
  Array* normals;       //!< The Array that handles the Normals.
  Array* vTexture;      //!< The Array that handles the VertexTextureCoordinates.

  
  Group* firstGroup;    //!< The first of all groups.
  Group* currentGroup;  //!< The currentGroup. this is the one we will work with.
  int groupCount;       //!< The Count of Groups.

  Material* material;   //!< Initial pointer to the Material. This can hold many materials, because Material can be added with Material::addMaterial(..)
  float scaleFactor;    //!< The Factor with which the Model should be scaled. \todo maybe one wants to scale the Model after Initialisation

  char* objPath;        //!< The Path where the obj and mtl-file are located.
  char* objFileName;    //!< The Name of the obj-file.
  char* mtlFileName;    //!< The Name of the mtl-file (parsed out of the obj-file)

  bool initialize (void);
  bool initGroup(Group* group);
  bool initFace (Face* face);
  bool cleanup(void);
  bool cleanupGroup(Group* group);
  bool cleanupFace(Face* face);
  bool cleanupFaceElement(FaceElement* faceElem);

  ///// readin ///
  bool importFile (char* fileName);
  bool readFromObjFile (void);
  
  bool readGroup (char* groupString);
  bool readVertex (char* vertexString);
  bool readFace (char* faceString);
  bool readVertexNormal (char* normalString);
  bool readVertexTexture (char* vTextureString);
  bool readMtlLib (char* matFile);
  bool readUseMtl (char* mtlString);

  bool importToGL (void);
  bool addGLElement (FaceElement* elem);

  bool buildVertexNormals ();

  void BoxModel (void);
};

#endif
