/* 
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#include "material.h"

/**
   \brief creates a default Material with no Name
   normally you call this to create a material List (for an obj-file) and then append with addMaterial()
*/
Material::Material()
{
  init();
  
  setName ("");
}

/**
   \brief creates a Material.
   \param mtlName Name of the Material to be added to the Material List
*/
Material::Material (char* mtlName)
{
  init();
  
  setName (mtlName);
}

/** 
    \brief deletes a Material
*/
Material::~Material()
{
  if (verbose >= 2)
    printf ("delete Material %s\n", name);
  if (nextMat != NULL)
    delete nextMat;
}

/**
   \brief adds a new Material to the List.
   this Function will append a new Material to the end of a Material List.
   \param mtlName The name of the Material to be added.
*/
Material* Material::addMaterial(char* mtlName)
{
  if (verbose >=2)
    printf ("adding Material %s\n", mtlName);
  Material* newMat = new Material(mtlName);
  Material* tmpMat = this;
  while (tmpMat->nextMat != NULL)
    {
      tmpMat = tmpMat->nextMat;
    }
      tmpMat->nextMat = newMat;
      return newMat;
  
}

/**
   \brief initializes a new Material with its default Values
*/
void Material::init(void)
{
  if (verbose >= 3)
    printf ("initializing new Material\n");
  nextMat = NULL;

  setIllum(1);
  setDiffuse(0,0,0);
  setAmbient(0,0,0);
  setSpecular(0,0,0);
  setShininess(2.0);
  setTransparency(0.0);
  
}

/**
   \brief Set the Name of the Material. (Important for searching)
   \param mtlName the Name of the Material to be set.
*/ 
void Material::setName (char* mtlName)
{
  if (verbose >= 3)
    printf("setting Material Name to %s", mtlName);
  strcpy(name, mtlName);
  //  printf ("adding new Material: %s, %p\n", this->getName(), this);

}
/**
   \returns The Name of The Material
*/
char* Material::getName (void)
{
  return name;
}

/**
   \brief Sets the Material Illumination Model. 
   \brief illu illumination Model in int form
*/
void Material::setIllum (int illum)
{
  if (verbose >= 3)
    printf("setting illumModel of Material %s to %i", name, illum);
  illumModel = illum;
  //  printf ("setting illumModel to: %i\n", illumModel);
}
/**
   \brief Sets the Material Illumination Model. 
   \brief illu illumination Model in char* form
*/void Material::setIllum (char* illum)
{
  setIllum (atoi(illum));
}

/**
   \brief Sets the Material Diffuse Color.
   \param r Red Color Channel.
   \param g Green Color Channel.
   \param b Blue Color Channel.
*/
void Material::setDiffuse (float r, float g, float b)
{
  if (verbose >= 3)
    printf ("setting Diffuse Color of Material %s to r=%f g=%f b=%f\n", name, r, g, b);
  diffuse[0] = r;
  diffuse[1] = g;
  diffuse[2] = b;  
  diffuse[3] = 1.0;

}
/**
   \brief Sets the Material Diffuse Color.
   \param rgb The red, green, blue channel in char format (with spaces between them)
*/
void Material::setDiffuse (char* rgb)
{
  char r[20],g[20],b[20];
  sscanf (rgb, "%s %s %s", r, g, b);
  setDiffuse (atof(r), atof(g), atof(b));
}

/**
   \brief Sets the Material Ambient Color. 
   \param r Red Color Channel.
   \param g Green Color Channel.
   \param b Blue Color Channel.
*/
void Material::setAmbient (float r, float g, float b)
{
  if (verbose >=3)
    printf ("setting Ambient Color of Material %s to r=%f g=%f b=%f\n", name, r, g, b);
  ambient[0] = r;
  ambient[1] = g;
  ambient[2] = b;
  ambient[3] = 1.0;
}
/**
   \brief Sets the Material Ambient Color.
   \param rgb The red, green, blue channel in char format (with spaces between them)
*/
void Material::setAmbient (char* rgb)
{
  char r[20],g[20],b[20];
  sscanf (rgb, "%s %s %s", r, g, b);
  setAmbient (atof(r), atof(g), atof(b));
}

/**
   \brief Sets the Material Specular Color. 
   \param r Red Color Channel.
   \param g Green Color Channel.
   \param b Blue Color Channel.
*/
void Material::setSpecular (float r, float g, float b)
{
  if (verbose >= 3)
    printf ("setting Specular Color of Material %s to r=%f g=%f b=%f\n", name, r, g, b);
  specular[0] = r;
  specular[1] = g;
  specular[2] = b;
  specular[3] = 1.0;
 }
/**
   \brief Sets the Material Specular Color.
   \param rgb The red, green, blue channel in char format (with spaces between them)
*/
void Material::setSpecular (char* rgb)
{
  char r[20],g[20],b[20];
  sscanf (rgb, "%s %s %s", r, g, b);
  setSpecular (atof(r), atof(g), atof(b));
}

/**
   \brief Sets the Material Shininess.
   \param shini stes the Shininess from float.
*/
void Material::setShininess (float shini)
{
  shininess = shini;
}
/**
   \brief Sets the Material Shininess.
   \param shini stes the Shininess from char*.
*/
void Material::setShininess (char* shini)
{
  setShininess (atof(shini));
}

/**
   \brief Sets the Material Transparency.
   \param trans stes the Transparency from int.
*/
void Material::setTransparency (float trans)
{
  if (verbose >= 3)
    printf ("setting Transparency of Material %s to %f\n", name, trans);
  transparency = trans;
}
/**
   \brief Sets the Material Transparency.
   \param trans stes the Transparency from char*.
*/
void Material::setTransparency (char* trans)
{
  char tr[20];
  sscanf (trans, "%s", tr);
  setTransparency (atof(tr));
}

/**
   \brief Search for a Material called mtlName
   \param mtlName the Name of the Material to search for
   \returns Material named mtlName if it is found. NULL otherwise.
*/
Material* Material::search (char* mtlName)
{
  if (verbose >=3)
    printf ("Searching for material %s", mtlName);
  Material* searcher = this;
  while (searcher != NULL)
    {
      if (verbose >= 3)
	printf (".");
      if (!strcmp (searcher->getName(), mtlName))
	{
	  if (verbose >= 3)
	    printf ("found.\n");
	  return searcher;
	}
      searcher = searcher->nextMat;
    }
  return NULL;
}

/**
   \brief sets the material with which the following Faces will be painted
*/
bool Material::select (void)
{
  // setting diffuse color
  //  glColor3f (diffuse[0], diffuse[1], diffuse[2]);
  glMaterialfv(GL_FRONT, GL_DIFFUSE, diffuse);

  // setting ambient color
  glMaterialfv(GL_FRONT, GL_AMBIENT, ambient);

  // setting up Sprecular
  glMaterialfv(GL_FRONT, GL_SPECULAR, specular);

  // setting up Shininess
  glMaterialf(GL_FRONT, GL_SHININESS, shininess);
  
  // setting illumination Model 
  if (illumModel == 1)
    glShadeModel(GL_FLAT);
  else if (illumModel >= 2)
    glShadeModel(GL_SMOOTH);
}
