/*! 
    \file world.h
    \brief Holds and manages all game data
*/ 

#ifndef WORLD_H
#define WORLD_H

#include "stdincl.h"
#include "story_entity.h"



class Track;
class WorldEntity;
class Camera;

//! The game environment
class World : public StoryEntity {

 public:
  World (char* name);
  World (int worldID);
  ~World ();

  template<typename T> 
    T* spawn(Location* loc, WorldEntity* owner);	// template to be able to spawn any derivation of WorldEntity
  template<typename T> 
    T* spawn(Placement* plc, WorldEntity* owner);
  
  virtual Error init();
  virtual Error start();
  virtual Error stop();
  virtual Error pause();
  virtual Error resume();

  virtual void load();

  void time_slice (Uint32 deltaT);
  void collide ();
  void draw ();
  void update ();	// maps Locations to Placements
  void calc_camera_pos (Location* loc, Placement* plc);
	
  void unload ();
  
  void setTrackLen(Uint32 tracklen);
  int getTrackLen();
  bool system_command (Command* cmd);
  Camera* getCamera();

  void spawn(WorldEntity* entity);
  void spawn(WorldEntity* entity, Location* loc);
  void spawn(WorldEntity* entity, Placement* plc);

  tList<WorldEntity>* entities;
  
  // base level data
  Track* track;
  Uint32 tracklen;   // number of Tracks the World consist of
  Vector* pathnodes;
  Camera* localCamera;  

 private:
  Uint32 lastFrame; //!> last time of frame
  bool bQuitOrxonox; //!> quit this application
  bool bQuitCurrentGame; //!> quit only the current game and return to menu
  bool bPause;

  char* worldName;
  int debugWorldNr;
  GLuint objectList;

  WorldEntity* localPlayer;
  
  void mainLoop();
  void synchronize();
  void handle_input();
  void time_slice();
  void collision();
  void display();
  void debug();
};

/**
    \brief spawn a new WorldEntity at a Location
    \param loc: the Location where the Entity should be spawned
    \param owner: a pointer to the parent of the Entity
    \return a pointer to the new WorldEntity or NULL if there was an error
    
    You can use this function to spawn any derivation of WorldEntity you want, just specify the desired
    class within the template specification brackets. Do not attempt to spawn any classes that have NOT been
    derived from WorldEntity, you won't even be able to compile the code. Note that this version of spawn()
    works with both free and bound WorldEntities.
*/ 
template<typename T> T* World::spawn(Location* loc = NULL, WorldEntity* owner = NULL)
{
  Location zeroloc;
  T* entity = new T();
  entities->add ((WorldEntity*)entity, LIST_ADD_NEXT);
  if( loc == NULL)
    {
      zeroloc.dist = 0;
      zeroloc.part = 0;
      zeroloc.pos = Vector();
      zeroloc.rot = Quaternion();
      loc = &zeroloc;
    }
  entity->init (loc, owner);
  if (entity->bFree)
    {
      track[loc->part].map_coords( loc, entity->get_placement());
    }
  entity->post_spawn ();
  return entity;
}

/**
    \brief spawn a new WorldEntity at a Placement
    \param lplc: the placement where the Entity should be spawned
    \param owner: a pointer to the parent of the Entity
    \return a pointer to the new WorldEntity or NULL if there was an error
    
    You can use this function to spawn any FREE derivation of WorldEntity you want, just specify the desired
    class within the template specification brackets. Do not attempt to spawn any classes that have NOT been
    derived from WorldEntity, you won't even be able to compile the code. Note that this version of spawn()
    works with free WorldEntities only, you will provoke an error message if you try to spawn a bound Entity with 
    a Placement.
*/ 
template<typename T> T* World::spawn(Placement* plc, WorldEntity* owner = NULL)
{
  T* entity = new T();
  entities->add ((WorldEntity*)entity, LIST_ADD_NEXT);
  entity->init (plc, owner);
  if (!entity->bFree)
    {
      printf("Can't spawn unfree entity with placement\n"); 
      entities->remove( (WorldEntity*)entity, LIST_FIND_FW);
      return NULL;
    }
  entity->post_spawn ();
  return entity;
}

#endif
