

/* 
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Christian Meyer
   co-programmer: ...
*/

#include "camera.h"
#include "world.h"
#include "world_entity.h"

using namespace std;

/**
   \brief creates a Camera
   
   This standard constructor sets all parameters to zero
*/
Camera::Camera ()
{
	bound = NULL;
}

/**
   \brief default destructor
*/
Camera::~Camera ()
{
}

/**
   \brief time based actualisation of camera parameters
   \param deltaT: The amount of time that has passed in milliseconds
   
   This is called by the World in every time_slice, use it to do fancy time dependant effects (such 
   as smooth camera movement or swaying).
*/
void Camera::time_slice (Uint32 deltaT)
{
	update_desired_place ();
	jump (NULL);
}

/**
  \brief this calculates the location where the track wants the camera to be
   
	This refreshes the placement the camera should have according to the bound entity's position on the track.
*/
void Camera::update_desired_place ()
{
		Orxonox *orx = Orxonox::getInstance();
		Location lookat;
		
		if( bound != NULL)
		{
			bound->get_lookat (&lookat);
			orx->get_world()->calc_camera_pos (&lookat, &desired_place);
		}
		else
		{
			desired_place.r = Vector (0,0,0);
			desired_place.w = Quaternion (); 
		}
}

/**
  \brief initialize rendering perspective according to this camera
   
	This is called immediately before the a rendering cycle starts, it sets all global rendering options as
	well as the GL_PROJECTION matrix according to the camera.
*/
void Camera::apply ()
{
	glMatrixMode (GL_PROJECTION);
  glLoadIdentity ();
		// view
		// TO DO: implement options for frustum generation
	glFrustum( -1.0,1.0,-1.0,1.0,1.5,250.0);
	glRotatef(h,1,0,0);
	glRotatef(h/2,0,1,0);
	glRotatef(h/4,0,0,1);
	h += 3;
//	glRotatef(90,0,0,1);
  	// rotation
  float matrix[4][4];
  actual_place.w.conjugate().matrix (matrix);
  glMultMatrixf ((float*)matrix);
  	// translation
  glTranslatef (-actual_place.r.x, -actual_place.r.y,- actual_place.r.z);
  
  glMatrixMode (GL_MODELVIEW);
  glLoadIdentity ();
  
  printf("Camera@%f/%f/%f\n",actual_place.r.x,actual_place.r.y,actual_place.r.z);
  Vector unique(1,0,0);
  Vector test = actual_place.w.apply (unique);
  printf("Camera&%f/%f/%f\n", test.x, test.y, test.z);
}

/**
  \brief set the camera position
  \param plc: The Placement to set the camera to 
	
	This will set the actual and desired placement of the camera to plc
*/
void Camera::jump (Placement* plc = NULL)
{
	if( plc == NULL)
	{
		actual_place = desired_place;
	}
	else
	{
		desired_place = *plc;
		actual_place = *plc;
	}
}

/**
  \brief bind the camera to an entity
  \param entity: The enitity to bind the camera to
	
	This sets the focus of the camera to the given entity. This means that it will use the given WorldEntity's
	Location and get_lookat() to determine the viewpoint the camera will render from.
	Note that you cannot bind a camera to a free entity.
*/
void Camera::bind (WorldEntity* entity)
{
	if( entity != NULL)
	{
		if( entity->isFree ()) printf("Cannot bind camera to free entity");
		else bound = entity;
	} 
}
