/* 
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Christian Meyer
   co-programmer: ...
*/


#include "command_node.h"
#include "keynames.h"
#include "ini_parser.h"

#include <stdio.h>
#include <strings.h>

using namespace std;

CommandNode::CommandNode (int ID)
{
	bound = new List<WorldEntity>();
	aliases = NULL;
	netID = ID;
	bLocalInput = false;
}

CommandNode::CommandNode (char* filename = DEFAULT_KEYBIND_FILE)
{
	bLocalInput = true;
	netID = 0;
	bound = new List<WorldEntity>();
	load_bindings (filename);
}

CommandNode::~CommandNode ()
{
	if( aliases != NULL) free (aliases);
	if( bound != NULL) delete bound;
}

void CommandNode::load_bindings (char* filename)
{
	FILE* stream;
	
	if( filename == NULL) filename = DEFAULT_KEYBIND_FILE;
	
		// remove old bindings if present
	if( aliases != NULL)
	{
		free (aliases);
		aliases = NULL;
	}
			
		// create parser
	IniParser parser (filename);
	if( parser.get_section ("Bindings") == -1)
	{
		printf("Could not find key bindings in %s\n", filename);
		return;
	}
			// allocate empty lookup table
	aliases = (KeyBindings*) calloc (1, sizeof (KeyBindings));
	
	char namebuf[256];
	char valuebuf[256];
	int* index;

	while( parser.next_var (namebuf, valuebuf) != -1)
	{
		index = name_to_index (namebuf);
		switch( index[0])
		{
			case 0:
				strcpy (aliases->keys[index[1]], valuebuf);
				break;
			case 1:
				strcpy (aliases->buttons[index[1]], valuebuf);
				break;
			default:
				break;
		}
	}
}

void CommandNode::bind (WorldEntity* entity)
{
	bound->add (entity, LIST_ADD_NEXT, true);
}

void CommandNode::unbind (WorldEntity* entity)
{
	bound->remove (entity, LIST_FIND_FW);
}

int* CommandNode::name_to_index (char* name)
{
	coord[0] = -1;
	if( (coord[1] = keyname_to_SDLK (name)) != -1) coord[0] = 0;
	if( (coord[1] = buttonname_to_SDLB (name)) != -1) coord[0] = 1;
	return coord;
}

void CommandNode::process ()
{
	if( bLocalInput) process_local ();
	else process_network ();
}

void CommandNode::process_local ()
{
	SDL_Event event;
	Command cmd;
	
	while( SDL_PollEvent (&event))
	{
		switch( event.type)
		{
			case SDL_KEYDOWN:
				strcpy (cmd.cmd, aliases->keys[event.key.keysym.sym]);
				cmd.bUp = false;
				if( strlen (cmd.cmd) > 0) relay (&cmd);
				break;
			case SDL_KEYUP:
				strcpy( cmd.cmd, aliases->keys[event.key.keysym.sym]);
				cmd.bUp = true;
				if( strlen (cmd.cmd) > 0) relay (&cmd);
				break;
			case SDL_MOUSEMOTION:
				strcpy( cmd.cmd, "cursor");
				cmd.x = event.motion.x;
				cmd.y = event.motion.y;
				cmd.xrel = event.motion.xrel;
				cmd.yrel = event.motion.yrel;
				break;
			case SDL_MOUSEBUTTONUP:
				strcpy( cmd.cmd, aliases->buttons[event.button.button]);
				cmd.bUp = true;
				if( strlen (cmd.cmd) > 0) relay (&cmd);
				break;
			case SDL_MOUSEBUTTONDOWN:
				strcpy( cmd.cmd, aliases->buttons[event.button.button]);
				cmd.bUp = false;
				if( strlen (cmd.cmd) > 0) relay (&cmd);
				break;
			case SDL_JOYAXISMOTION:
			case SDL_JOYBALLMOTION:
			case SDL_JOYHATMOTION:
			case SDL_JOYBUTTONDOWN:
			case SDL_JOYBUTTONUP:
				break;
			default:
					// TO DO: Send other events to orxonox sysevent handler
				break;
		}
	}
}

void CommandNode::process_network ()
{

}

void CommandNode::relay (Command* cmd)
{
	List<WorldEntity>* plist = bound;
	
	while( (plist = plist->get_next()) != NULL)
	{
		(plist->get_object())->command (cmd);
	}
}
