
#ifndef VECTOR_H
#define VECTOR_H

#include <math.h>
#define PI 3.14159265359f

/**
  Class for 3-dimensional vector calculation
  
  Supports all common vector operations (dot, cross, lenght and so on)
*/
class Vector {

  public:
 
  float x, y, z;

  Vector (float x, float y, float z) : x(x), y(y), z(z) {}
  Vector () : x(0), y(0), z(0) {}
  ~Vector () {}

  Vector operator+ (const Vector& v) const;
  Vector operator- (const Vector& v) const;
  float operator* (const Vector& v) const;
  Vector operator* (float f) const;
  Vector operator/ (float f) const;
  float dot (const Vector& v) const;
  Vector cross (const Vector& v) const;
  float len() const;
  void normalize();
};

float angle_deg (const Vector& v1, const Vector& v2);
float angle_rad (const Vector& v1, const Vector& v2);

/**
  Class to handle 3-dimensional rotations
  
  Can create a rotation from several inputs, currently stores rotation using a 3x3 Matrix
*/
class Rotation {
  public:
  
  float m[9];
  
  Rotation ( const Vector& v);
  Rotation ( const Vector& axis, float angle);
  Rotation ( float pitch, float yaw, float roll);
  Rotation ();
  ~Rotation () {}
  
};

Vector rotate_vector( const Vector& v, const Rotation& r);

/**
  Class to store Lines in 3-dimensional space

  Supports line-to-line distance measurements and rotation
*/
class Line
{
  public:
  
  Vector r, a;
  
  Line ( Vector r, Vector a) : r(r), a(a) {}
  Line () : r(Vector(0,0,0)), a(Vector (1,1,1)) {}
  ~Line () {}
  
  float distance (const Line& l) const;
  float distance_point (const Vector& v) const;
  Vector* footpoints (const Line& l) const;
  float len () const;
  
  void rotate(const Rotation& rot);
};

/**
  Class to handle planes in 3-dimensional space

  Critical for polygon-based collision detection
*/
class Plane
{
  public:
  
  Vector n;
  float k;
  
  Plane (Vector a, Vector b, Vector c);
  Plane (Vector norm, Vector p);
  Plane (Vector n, float k) : n(n), k(k) {}
  Plane () : n(Vector(1,1,1)), k(0) {}
  ~Plane () {}
  
  Vector intersect_line (const Line& l) const;
  float distance_point (const Vector& p) const;
  float locate_point (const Vector& p) const;
};

#endif
