
#ifndef COLLISION_H
#define COLLISION_H

#include "vector.h"
#include "coordinates.h"
#include <stdlib.h>
#include <stdio.h>
#include <strings.h>

typedef struct CC_Tree
{
  unsigned long n;
  union
  {
  struct CC_Tree** b;
  unsigned long ID;
  } data;
  float r;
  Vector m;
} CC_Tree;

/**
  This class implements a more or less efficient collision system based on nested hitzones.
  Instead of using a brute force approach (try if any hitzone intersects with any other hitzone)
  this features a tree of spheric hitzones. Only some of them are actually the solid groundstones
  the collision model bases on, the others serve to group them into sets of spheres that are only
  checked for collision when the assigned top level sphere has registered a collision, preventing
  unnessessary checks (like between every sphere of two collision clusters at the other end of the world)
  from being performed.
  The CollisionCluster features collision detection between multiple CollisionClusters as well as
  traceing a collision between a line of defined length and a cluster. In both cases the base spheres
  that have intersected are marked with a flag in an unsigned long for hitlocation queries. In the case
  of a trace, the exact point of interception is returned as well.
*/

class CollisionCluster {
  
  CC_Tree* root;
  
  
 public:
  CollisionCluster (float r, Vector m); // simple cluster
  CollisionCluster (char* filename); // get cluster from file
  ~CollisionCluster ();
  
  int store (char* filename);
  
  friend bool cctree_trace( const Placement* p, CC_Tree* t, unsigned long* hitflags, const Line* trace, Vector* impactpoint);
  friend bool cctree_iterate(const Placement* pa, CC_Tree* ta, unsigned long* ahitflags, const Placement* pb, CC_Tree* tb, unsigned long* bhitflags);
  friend bool check_trace (const Placement* pa, const CollisionCluster* a, unsigned long* ahitflags, const Line* trace, Vector* impactpoint);
  friend bool check_collision (const Placement* pa, const CollisionCluster* a, unsigned long* ahitflags, const Placement* pb, const CollisionCluster* b, unsigned long* bhitflags);
};

bool sphere_sphere_collision( Vector m1, float r1, Vector m2, float r2);
bool trace_sphere_collision( Vector m, float r, const Line* l, Vector* impactpoint);

void setflag( unsigned long* flags, unsigned long ID);

void free_CC_Tree( CC_Tree* tree);
CC_Tree* load_CC_Tree (FILE* stream);
int save_CC_Tree (CC_Tree* tree, FILE* stream);

#endif
