/*
 *   ORXONOX - the hottest 3D action shooter ever to exist
 *                    > www.orxonox.net <
 *
 *
 *   License notice:
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation; either version 2
 *   of the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 *   Author:
 *      Fabian 'x3n' Landau
 *   Co-authors:
 *      ...
 *
 */

#ifndef _CommandExecutor_H__
#define _CommandExecutor_H__

#include "CorePrereqs.h"

#include <map>

#include "CommandEvaluation.h"

#define COMMAND_EXECUTOR_CURSOR "$"

namespace orxonox
{
    class _CoreExport CommandExecutor
    {
        public:
            static bool execute(const std::string& command, bool useTcl = true);
            static std::string complete(const std::string& command);
            static std::string hint(const std::string& command);

            static CommandEvaluation evaluate(const std::string& command);
            static const CommandEvaluation& getLastEvaluation();

            static ConsoleCommand& addConsoleCommandShortcut(ConsoleCommand* command);
            static ConsoleCommand* getConsoleCommandShortcut(const std::string& name);
            static ConsoleCommand* getLowercaseConsoleCommandShortcut(const std::string& name);

            /** @brief Returns the map that stores all console commands. @return The const_iterator */
            static inline const std::map<std::string, ConsoleCommand*>& getConsoleCommandShortcutMap() { return CommandExecutor::getInstance().consoleCommandShortcuts_; }
            /** @brief Returns a const_iterator to the beginning of the map that stores all console commands. @return The const_iterator */
            static inline std::map<std::string, ConsoleCommand*>::const_iterator getConsoleCommandShortcutMapBegin() { return CommandExecutor::getInstance().consoleCommandShortcuts_.begin(); }
            /** @brief Returns a const_iterator to the end of the map that stores all console commands. @return The const_iterator */
            static inline std::map<std::string, ConsoleCommand*>::const_iterator getConsoleCommandShortcutMapEnd() { return CommandExecutor::getInstance().consoleCommandShortcuts_.end(); }

            /** @brief Returns the map that stores all console commands with their names in lowercase. @return The const_iterator */
            static inline const std::map<std::string, ConsoleCommand*>& getLowercaseConsoleCommandShortcutMap() { return CommandExecutor::getInstance().consoleCommandShortcuts_LC_; }
            /** @brief Returns a const_iterator to the beginning of the map that stores all console commands with their names in lowercase. @return The const_iterator */
            static inline std::map<std::string, ConsoleCommand*>::const_iterator getLowercaseConsoleCommandShortcutMapBegin() { return CommandExecutor::getInstance().consoleCommandShortcuts_LC_.begin(); }
            /** @brief Returns a const_iterator to the end of the map that stores all console commands with their names in lowercase. @return The const_iterator */
            static inline std::map<std::string, ConsoleCommand*>::const_iterator getLowercaseConsoleCommandShortcutMapEnd() { return CommandExecutor::getInstance().consoleCommandShortcuts_LC_.end(); }

        private:
            CommandExecutor() {}
            CommandExecutor(const CommandExecutor& other);
            ~CommandExecutor() {}

            static CommandExecutor& getInstance();
            static CommandEvaluation& getEvaluation();

            static void parse(const std::string& command, bool bInitialize = true);

            static bool argumentsFinished(unsigned int num);
            static unsigned int argumentsFinished();
            static unsigned int argumentsGiven();
            static bool enoughArgumentsGiven(ConsoleCommand* command, unsigned int head);
            static std::string getToken(unsigned int index);

            static void createListOfPossibleFunctionClasses(const std::string& fragment);
            static void createListOfPossibleFunctions(const std::string& fragment, Identifier* identifier = 0);

            static Identifier* getPossibleIdentifier(const std::string& name);
            static ConsoleCommand* getPossibleCommand(const std::string& name, Identifier* identifier = 0);

            static bool compareStringsInList(const std::pair<const std::string*, const std::string*>& first, const std::pair<const std::string*, const std::string*>& second);


            CommandEvaluation evaluation_;
            std::map<std::string, ConsoleCommand*> consoleCommandShortcuts_;
            std::map<std::string, ConsoleCommand*> consoleCommandShortcuts_LC_;
    };
}

#endif /* _CommandExecutor_H__ */
