/*
 *   ORXONOX - the hottest 3D action shooter ever to exist
 *                    > www.orxonox.net <
 *
 *
 *   License notice:
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation; either version 2
 *   of the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 *   Author:
 *      Fabian 'x3n' Landau
 *   Co-authors:
 *      Benjamin Knecht
 *
 */

#include "OrxonoxStableHeaders.h"
#include "SpaceShip.h"

#include <string>

#include <OgreCamera.h>
#include <OgreRenderWindow.h>
#include <OgreParticleSystem.h>
#include <OgreSceneNode.h>

#include "CameraHandler.h"
#include "tinyxml/tinyxml.h"
#include "ois/OIS.h"
#include "util/Convert.h"
#include "util/Math.h"
#include "core/CoreIncludes.h"
#include "core/ConfigValueIncludes.h"
#include "core/Debug.h"
#include "GraphicsEngine.h"
#include "core/InputManager.h"
#include "particle/ParticleInterface.h"
#include "Projectile.h"
#include "core/XMLPort.h"
#include "core/ConsoleCommand.h"
#include "network/Client.h"

namespace orxonox
{
    ConsoleCommand(SpaceShip, setMaxSpeedTest, AccessLevel::Debug, false);
    ConsoleCommandGeneric(test1, SpaceShip, createExecutor(createFunctor(&SpaceShip::setMaxSpeedTest), "setMaxSpeed", AccessLevel::Debug), false);
    ConsoleCommandGeneric(test2, SpaceShip, createExecutor(createFunctor(&SpaceShip::setMaxSpeedTest), "setMaxBlubber", AccessLevel::Debug), false);
    ConsoleCommandGeneric(test3, SpaceShip, createExecutor(createFunctor(&SpaceShip::setMaxSpeedTest), "setRofl", AccessLevel::Debug), false);

    CreateFactory(SpaceShip);

    SpaceShip* SpaceShip::instance_s;

    SpaceShip::SpaceShip() :
      //testvector_(0,0,0),
      //bInvertYAxis_(false),
      setMouseEventCallback_(false),
      bLMousePressed_(false),
      bRMousePressed_(false),
      camNode_(0),
      cam_(0),
      camName_("CamNode"),
      tt_(0),
      redNode_(0),
      greenNode_(0),
      blinkTime_(0.0f),
      chNearNode_(0),
      chFarNode_(0),
      timeToReload_(0.0f),
      //reloadTime_(0.0f),
      maxSideAndBackSpeed_(0.0f),
      maxSpeed_(0.0f),
      maxRotation_(0.0f),
      translationAcceleration_(0.0f),
      rotationAcceleration_(0.0f),
      translationDamping_(0.0f),
      rotationDamping_(0.0f),
      maxRotationRadian_(0),
      rotationAccelerationRadian_(0),
      rotationDampingRadian_(0),
      zeroRadian_(0),
      mouseXRotation_(0),
      mouseYRotation_(0),
      mouseX_(0.0f),
      mouseY_(0.0f),
      emitterRate_(0.0f),
      server_(false)
    {
        RegisterObject(SpaceShip);
        this->registerAllVariables();

        SpaceShip::instance_s = this;

        this->setConfigValues();

        this->setRotationAxis(1, 0, 0);
        this->setStatic(false);

        COUT(3) << "Info: SpaceShip was loaded" << std::endl;
    }

    SpaceShip::~SpaceShip()
    {
        if (this->tt_)
            delete this->tt_;
        if(setMouseEventCallback_)
          InputManager::removeMouseHandler("SpaceShip");
    }

    bool SpaceShip::create(){
      if(Model::create())
        this->init();
      else
        return false;
      return true;
    }

    void SpaceShip::registerAllVariables(){
      registerVar( &camName_, camName_.length()+1, network::STRING, 0x1);
      registerVar( &maxSpeed_, sizeof(maxSpeed_), network::DATA, 0x1);
      registerVar( &maxSideAndBackSpeed_, sizeof(maxSideAndBackSpeed_), network::DATA, 0x1);
      registerVar( &maxRotation_, sizeof(maxRotation_), network::DATA, 0x1);
      registerVar( &translationAcceleration_, sizeof(translationAcceleration_), network::DATA, 0x1);
      registerVar( &rotationAcceleration_, sizeof(rotationAcceleration_), network::DATA, 0x1);
      registerVar( &rotationAccelerationRadian_, sizeof(rotationAccelerationRadian_), network::DATA, 0x1);
      registerVar( &translationDamping_, sizeof(translationDamping_), network::DATA, 0x1);
      registerVar( &rotationDamping_, sizeof(rotationDamping_), network::DATA, 0x1);
      registerVar( &rotationDampingRadian_, sizeof(rotationDampingRadian_), network::DATA, 0x1);

    }

    void SpaceShip::init()
    {
    if ((server_ || ( network::Client::getSingleton() && network::Client::getSingleton()->getShipID()==objectID ) ))
    {
          if (!setMouseEventCallback_)
          {
              InputManager::addMouseHandler(this, "SpaceShip");
              setMouseEventCallback_ = true;
          }
    }

        // START CREATING THRUSTER
        this->tt_ = new ParticleInterface(GraphicsEngine::getSingleton().getSceneManager(),"twinthruster" + this->getName(),"Orxonox/engineglow");
        this->tt_->getParticleSystem()->setParameter("local_space","true");
        this->tt_->newEmitter();
/*
        this->tt_->setDirection(Vector3(0,0,1));
        this->tt_->setPositionOfEmitter(0, Vector3(20,-1,-15));
        this->tt_->setPositionOfEmitter(1, Vector3(-20,-1,-15));
*/
        this->tt_->setDirection(Vector3(-1,0,0));
        this->tt_->setPositionOfEmitter(0, Vector3(-15,20,-1));
        this->tt_->setPositionOfEmitter(1, Vector3(-15,-20,-1));
        this->tt_->setVelocity(50);

        emitterRate_ = tt_->getRate();

        Ogre::SceneNode* node2 = this->getNode()->createChildSceneNode(this->getName() + "particle2");
        node2->setInheritScale(false);
        tt_->addToSceneNode(node2);
        // END CREATING THRUSTER

        // START CREATING BLINKING LIGHTS
        this->redBillboard_.setBillboardSet("Examples/Flare", ColourValue(1.0, 0.0, 0.0), 1);
        this->greenBillboard_.setBillboardSet("Examples/Flare", ColourValue(0.0, 1.0, 0.0), 1);

        this->redNode_ = this->getNode()->createChildSceneNode(this->getName() + "red", Vector3(0.3, 4.0, -0.3));
        this->redNode_->setInheritScale(false);
        this->greenNode_ = this->getNode()->createChildSceneNode(this->getName() + "green", Vector3(0.3, -4.0, -0.3));
        this->greenNode_->setInheritScale(false);

        this->redNode_->attachObject(this->redBillboard_.getBillboardSet());
        this->redNode_->setScale(0.3, 0.3, 0.3);

        this->greenNode_->attachObject(this->greenBillboard_.getBillboardSet());
        this->greenNode_->setScale(0.3, 0.3, 0.3);
        // END CREATING BLINKING LIGHTS

        // START of testing crosshair
        this->crosshairNear_.setBillboardSet("Orxonox/Crosshair", ColourValue(1.0, 1.0, 0.0), 1);
        this->crosshairFar_.setBillboardSet("Orxonox/Crosshair", ColourValue(1.0, 1.0, 0.0), 1);

        this->chNearNode_ = this->getNode()->createChildSceneNode(this->getName() + "near", Vector3(50.0, 0.0, 0.0));
        this->chNearNode_->setInheritScale(false);
        this->chFarNode_ = this->getNode()->createChildSceneNode(this->getName() + "far", Vector3(200.0, 0.0, 0.0));
        this->chFarNode_->setInheritScale(false);

        this->chNearNode_->attachObject(this->crosshairNear_.getBillboardSet());
        this->chNearNode_->setScale(0.2, 0.2, 0.2);

        this->chFarNode_->attachObject(this->crosshairFar_.getBillboardSet());
        this->chFarNode_->setScale(0.4, 0.4, 0.4);

        createCamera();
        // END of testing crosshair
    }

    void SpaceShip::setConfigValues()
    {
        SetConfigValue(bInvertYAxis_, false).description("Set this to true for joystick-like mouse behaviour (mouse up = ship down).");
        SetConfigValue(reloadTime_, 0.125).description("The reload time of the weapon in seconds");
        SetConfigValue(testvector_, Vector3()).description("asdfblah");
    }

    void SpaceShip::loadParams(TiXmlElement* xmlElem)
    {
        Model::loadParams(xmlElem);
        this->create();
        this->getFocus();
/*
        if (xmlElem->Attribute("forward") && xmlElem->Attribute("rotateupdown") && xmlElem->Attribute("rotaterightleft") && xmlElem->Attribute("looprightleft"))
        {
            std::string forwardStr = xmlElem->Attribute("forward");
            std::string rotateupdownStr = xmlElem->Attribute("rotateupdown");
            std::string rotaterightleftStr = xmlElem->Attribute("rotaterightleft");
            std::string looprightleftStr = xmlElem->Attribute("looprightleft");

            String2Number<float>(this->maxSpeedForward_, forwardStr);
            String2Number<float>(this->maxSpeedRotateUpDown_, rotateupdownStr);
            String2Number<float>(this->maxSpeedRotateRightLeft_, rotaterightleftStr);
            String2Number<float>(this->maxSpeedLoopRightLeft_, looprightleftStr);

            COUT(4) << "Loader: Initialized spaceship steering with values " << maxSpeedForward_ << " " << maxSpeedRotateUpDown_ << " " << maxSpeedRotateRightLeft_ << " " << maxSpeedLoopRightLeft_ << " " << std::endl;
      }
*/
        if (xmlElem->Attribute("maxSpeed") && xmlElem->Attribute("maxSideAndBackSpeed") && xmlElem->Attribute("maxRotation") && xmlElem->Attribute("transAcc") && xmlElem->Attribute("rotAcc") && xmlElem->Attribute("transDamp") && xmlElem->Attribute("rotDamp"))
        {

            std::string msStr = xmlElem->Attribute("maxSpeed");
            std::string msabsStr = xmlElem->Attribute("maxSideAndBackSpeed");
            std::string mrStr = xmlElem->Attribute("maxRotation");
            std::string taStr = xmlElem->Attribute("transAcc");
            std::string raStr = xmlElem->Attribute("rotAcc");
            std::string tdStr = xmlElem->Attribute("transDamp");
            std::string rdStr = xmlElem->Attribute("rotDamp");

            convertValue<std::string, float>(&this->maxSpeed_, msStr);
            convertValue<std::string, float>(&this->maxSideAndBackSpeed_, msabsStr);
            convertValue<std::string, float>(&this->maxRotation_, mrStr);
            convertValue<std::string, float>(&this->translationAcceleration_, taStr);
            convertValue<std::string, float>(&this->rotationAcceleration_, raStr);
            convertValue<std::string, float>(&this->translationDamping_, tdStr);
            convertValue<std::string, float>(&this->rotationDamping_, rdStr);

            this->maxRotationRadian_ = Radian(this->maxRotation_);
            this->rotationAccelerationRadian_ = Radian(this->rotationAcceleration_);
            this->rotationDampingRadian_ = Radian(this->rotationDamping_);

            COUT(4) << "Loader: Initialized SpaceShip" << std::endl;
        }

        if (xmlElem->Attribute("camera"))
        {
            this->setCamera();
        }
    }

    void SpaceShip::setCamera(const std::string& camera)
    {
      camName_=camera;
      // change camera attributes here, if you want to ;)
    }
    
    void SpaceShip::getFocus(){
      COUT(4) << "requesting focus" << std::endl;
      if(network::Client::getSingleton()==0 || network::Client::getSingleton()->getShipID()==objectID)
        CameraHandler::getInstance()->requestFocus(cam_);
      
    }
    
    void SpaceShip::createCamera(){
//       COUT(4) << "begin camera creation" << std::endl;
      this->camNode_ = this->getNode()->createChildSceneNode(camName_);
      COUT(4) << "position: (this)" << this->getNode()->getPosition() << std::endl;
      this->camNode_->setPosition(/*this->getNode()->getPosition() +*/ Vector3(-50,0,10));
      COUT(4) << "position: (cam)" << this->camNode_->getPosition() << std::endl;
/*
//        node->setInheritOrientation(false);
        cam->setPosition(Vector3(0,50,-150));
        cam->lookAt(Vector3(0,20,0));
        cam->roll(Degree(0));
      *//*COUT(4) << "creating new camera" << std::endl;*/
      cam_ = new Camera(this->camNode_);

      cam_->setTargetNode(this->getNode());
//        cam->setPosition(Vector3(0,-350,0));
      if(network::Client::getSingleton()!=0 && network::Client::getSingleton()->getShipID()==objectID){
        this->setBacksync(true);
        CameraHandler::getInstance()->requestFocus(cam_);
      }

    }

    void SpaceShip::setMaxSpeed(float value)
    { this->maxSpeed_ = value; }
    void SpaceShip::setMaxSideAndBackSpeed(float value)
    { this->maxSideAndBackSpeed_ = value; }
    void SpaceShip::setMaxRotation(float value)
    { this->maxRotation_ = value; this->maxRotationRadian_ = Radian(value); }
    void SpaceShip::setTransAcc(float value)
    { this->translationAcceleration_ = value; }
    void SpaceShip::setRotAcc(float value)
    { this->rotationAcceleration_ = value; this->rotationAccelerationRadian_ = Radian(value); }
    void SpaceShip::setTransDamp(float value)
    { this->translationDamping_ = value; }
    void SpaceShip::setRotDamp(float value)
    { this->rotationDamping_ = value; this->rotationDampingRadian_ = Radian(value); }

    /**
        @brief XML loading and saving.
        @param xmlelement The XML-element
        @param loading Loading (true) or saving (false)
        @return The XML-element
    */
    void SpaceShip::XMLPort(Element& xmlelement, XMLPort::Mode mode)
    {
        Model::XMLPort(xmlelement, mode);

        XMLPortParamLoadOnly(SpaceShip, "camera", setCamera, xmlelement, mode);
        XMLPortParamLoadOnly(SpaceShip, "maxSpeed", setMaxSpeed, xmlelement, mode);
        XMLPortParamLoadOnly(SpaceShip, "maxSideAndBackSpeed", setMaxSideAndBackSpeed, xmlelement, mode);
        XMLPortParamLoadOnly(SpaceShip, "maxRotation", setMaxRotation, xmlelement, mode);
        XMLPortParamLoadOnly(SpaceShip, "transAcc", setTransAcc, xmlelement, mode);
        XMLPortParamLoadOnly(SpaceShip, "rotAcc", setRotAcc, xmlelement, mode);
        XMLPortParamLoadOnly(SpaceShip, "transDamp", setTransDamp, xmlelement, mode);
        XMLPortParamLoadOnly(SpaceShip, "rotDamp", setRotDamp, xmlelement, mode);
        server_=true; // TODO: this is only a hack
        SpaceShip::create();
        getFocus();
    }

    int sgn(float x)
    {
        if (x >= 0)
            return 1;
        else
            return -1;
    }

    bool SpaceShip::mouseMoved(const MouseState& state)
    {
/*
        this->mouseX += e.state.X.rel;
        if (this->bInvertMouse_)
            this->mouseY += e.state.Y.rel;
        else
            this->mouseY -= e.state.Y.rel;

//        if(mouseX>maxMouseX) maxMouseX = mouseX;
//        if(mouseX<minMouseX) minMouseX = mouseX;
//        cout << "mouseX: " << mouseX << "\tmouseY: " << mouseY << endl;

        this->moved = true;
*/
        if (this->bRMousePressed_)
        {
            this->camNode_->roll(Degree(-state.X.rel * 0.10));
            this->camNode_->yaw(Degree(state.Y.rel * 0.10));
        }
        else
        {
            float minDimension = state.height;
            if (state.width < minDimension)
                minDimension = state.width;

            this->mouseX_ += state.X.rel;
            if (this->mouseX_ < -minDimension)
                this->mouseX_ = -minDimension;
            if (this->mouseX_ > minDimension)
                this->mouseX_ = minDimension;

            this->mouseY_ += state.Y.rel;
            if (this->mouseY_ < -minDimension)
                this->mouseY_ = -minDimension;
            if (this->mouseY_ > minDimension)
                this->mouseY_ = minDimension;

            float xRotation = this->mouseX_ / minDimension;
            xRotation = xRotation*xRotation * sgn(xRotation);
            xRotation *= -this->rotationAcceleration_;
            if (xRotation > this->maxRotation_)
                xRotation = this->maxRotation_;
            if (xRotation < -this->maxRotation_)
                xRotation = -this->maxRotation_;
            this->mouseXRotation_ = Radian(xRotation);

            float yRotation = this->mouseY_ / minDimension;
            yRotation = yRotation*yRotation * sgn(yRotation);
            yRotation *= this->rotationAcceleration_;
            if (yRotation > this->maxRotation_)
                yRotation = this->maxRotation_;
            if (yRotation < -this->maxRotation_)
                yRotation = -this->maxRotation_;
            this->mouseYRotation_ = Radian(yRotation);
        }

        return true;
    }

    bool SpaceShip::mouseButtonPressed(const MouseState& state, MouseButton::Enum id)
    {
        if (id == MouseButton::Left)
            this->bLMousePressed_ = true;
        else if (id == MouseButton::Right)
            this->bRMousePressed_ = true;

        return true;
    }

    bool SpaceShip::mouseButtonReleased(const MouseState& state, MouseButton::Enum id)
    {
        if (id == MouseButton::Left)
            this->bLMousePressed_ = false;
        else if (id == MouseButton::Right)
        {
            this->bRMousePressed_ = false;
            this->camNode_->resetOrientation();
        }

        return true;
    }

    void SpaceShip::tick(float dt)
    {
        if (this->redNode_ && this->greenNode_)
        {
            this->blinkTime_ += dt;
            float redScale = 0.15 + 0.15 * sin(this->blinkTime_ * 10.0);
            float greenScale = 0.15 - 0.15 * sin(this->blinkTime_ * 10.0);
            this->redNode_->setScale(redScale, redScale, redScale);
            this->greenNode_->setScale(greenScale, greenScale, greenScale);
        }

        if (this->timeToReload_ > 0)
            this->timeToReload_ -= dt;
        else
            this->timeToReload_ = 0;

        if (this->bLMousePressed_ && this->timeToReload_ <= 0)
        {
            Projectile *p = new Projectile(this);
            p->setBacksync(true);
            this->timeToReload_ = this->reloadTime_;
        }


        // #####################################
        // ############# STEERING ##############
        // #####################################

        if (this->velocity_.x > this->maxSpeed_)
            this->velocity_.x = this->maxSpeed_;
        if (this->velocity_.x < -this->maxSideAndBackSpeed_)
            this->velocity_.x = -this->maxSideAndBackSpeed_;
        if (this->velocity_.y > this->maxSideAndBackSpeed_)
            this->velocity_.y = this->maxSideAndBackSpeed_;
        if (this->velocity_.y < -this->maxSideAndBackSpeed_)
            this->velocity_.y = -this->maxSideAndBackSpeed_;
        if (this->rotationRate_ > this->maxRotationRadian_)
            this->rotationRate_ = this->maxRotationRadian_;
        if (this->rotationRate_ < -this->maxRotationRadian_)
            this->rotationRate_ = -this->maxRotationRadian_;

        if (this->acceleration_.x == 0)
        {
            if (this->velocity_.x > 0)
            {
                this->velocity_.x -= (this->translationDamping_ * dt);
                if (this->velocity_.x < 0)
                    this->velocity_.x = 0;
            }
            else if (this->velocity_.x < 0)
            {
                this->velocity_.x += (this->translationDamping_ * dt);
                if (this->velocity_.x > 0)
                    this->velocity_.x = 0;
            }
        }

        if (this->acceleration_.y == 0)
        {
            if (this->velocity_.y > 0)
            {
                this->velocity_.y -= (this->translationDamping_ * dt);
                if (this->velocity_.y < 0)
                    this->velocity_.y = 0;
            }
            else if (this->velocity_.y < 0)
            {
                this->velocity_.y += (this->translationDamping_ * dt);
                if (this->velocity_.y > 0)
                    this->velocity_.y = 0;
            }
        }

        if (this->momentum_ == this->zeroRadian_)
        {
            if (this->rotationRate_ > this->zeroRadian_)
            {
                this->rotationRate_ -= (this->rotationDampingRadian_ * dt);
                if (this->rotationRate_ < this->zeroRadian_)
                    this->rotationRate_ = 0;
            }
            else if (this->rotationRate_ < this->zeroRadian_)
            {
                this->rotationRate_ += (this->rotationDampingRadian_ * dt);
                if (this->rotationRate_ > this->zeroRadian_)
                    this->rotationRate_ = 0;
            }
        }

        if( (network::Client::getSingleton() &&  network::Client::getSingleton()->getShipID() == objectID) || server_ )
        {
          COUT(4) << "steering our ship: " << objectID << std::endl;
          if (InputManager::isKeyDown(KeyCode::Up) || InputManager::isKeyDown(KeyCode::W))
            this->acceleration_.x = this->translationAcceleration_;
          else if(InputManager::isKeyDown(KeyCode::Down) || InputManager::isKeyDown(KeyCode::S))
            this->acceleration_.x = -this->translationAcceleration_;
          else
            this->acceleration_.x = 0;

          if (InputManager::isKeyDown(KeyCode::Right) || InputManager::isKeyDown(KeyCode::D))
            this->acceleration_.y = -this->translationAcceleration_;
          else if (InputManager::isKeyDown(KeyCode::Left) || InputManager::isKeyDown(KeyCode::A))
            this->acceleration_.y = this->translationAcceleration_;
          else
            this->acceleration_.y = 0;

          if (InputManager::isKeyDown(KeyCode::Delete) || InputManager::isKeyDown(KeyCode::Q))
            this->momentum_ = Radian(-this->rotationAccelerationRadian_);
          else if (InputManager::isKeyDown(KeyCode::PageDown) || InputManager::isKeyDown(KeyCode::E))
            this->momentum_ = Radian(this->rotationAccelerationRadian_);
          else
            this->momentum_ = 0;
        }/*else
          COUT(4) << "not steering ship: " << objectID << " our ship: " << network::Client::getSingleton()->getShipID() << std::endl;*/

        WorldEntity::tick(dt);

        this->roll(this->mouseXRotation_ * dt);
        if (this->bInvertYAxis_)
            this->yaw(Radian(-this->mouseYRotation_ * dt));
        else
            this->yaw(Radian(this->mouseYRotation_ * dt));

        if (this->acceleration_.x > 0)
            this->tt_->setRate(emitterRate_);
        else
            this->tt_->setRate(0);

/*
        if (mKeyboard->isKeyDown(OIS::KC_UP) || mKeyboard->isKeyDown(OIS::KC_W))
            this->moveForward(speed);
        else
            this->moveForward(0);

        if(mKeyboard->isKeyDown(OIS::KC_DOWN) || mKeyboard->isKeyDown(OIS::KC_S))
            this->brakeForward(speed);
        else
            this->brakeForward(speed/10);

        if (mKeyboard->isKeyDown(OIS::KC_RIGHT) || mKeyboard->isKeyDown(OIS::KC_D))
            this->loopRight(loop);
        else
            this->loopRight(0);

        if (mKeyboard->isKeyDown(OIS::KC_LEFT) || mKeyboard->isKeyDown(OIS::KC_A))
            this->loopLeft(loop);
        else
            this->loopLeft(0);

        if(moved)
        {
            if (mouseY<=0)
                this->rotateUp(-mouseY*rotate);
            if (mouseY>0)
                this->rotateDown(mouseY*rotate);
            if (mouseX>0)
                this->rotateRight(mouseX*rotate);
            if (mouseX<=0)
                this->rotateLeft(-mouseX*rotate);

            mouseY = 0;
            mouseX = 0;
            moved = false;
        }*/
/*        else
        {
            this->rotateUp(0);
            this->rotateDown(0);
            this->rotateRight(0);
            this->rotateLeft(0);
        }*/
/*
        if(moveForward_ > 0)
        {
            accelerationForward_ = moveForward_;
            if(speedForward_ < maxSpeedForward_)
                speedForward_ += accelerationForward_*dt;
            if(speedForward_ > maxSpeedForward_)
                speedForward_ = maxSpeedForward_;
        }

        if(moveForward_ <= 0)
        {
            accelerationForward_ = -brakeForward_;
            if(speedForward_ > 0)
                speedForward_ += accelerationForward_*dt;
            if(speedForward_ < 0)
                speedForward_ = 0;
        }

        if(rotateUp_ > 0)
        {
            accelerationRotateUpDown_ = rotateUp_;
            if(speedRotateUpDown_ < maxSpeedRotateUpDown_)
                speedRotateUpDown_ += accelerationRotateUpDown_*dt;
            if(speedRotateUpDown_ > maxSpeedRotateUpDown_)
            speedRotateUpDown_ = maxSpeedRotateUpDown_;
        }

        if(rotateDown_ > 0)
        {
            accelerationRotateUpDown_ = rotateDown_;
            if(speedRotateUpDown_ > -maxSpeedRotateUpDown_)
                speedRotateUpDown_ -= accelerationRotateUpDown_*dt;
            if(speedRotateUpDown_ < -maxSpeedRotateUpDown_)
                speedRotateUpDown_ = -maxSpeedRotateUpDown_;
        }

        if(rotateUp_ == 0 && rotateDown_ == 0)
        {
            accelerationRotateUpDown_ = brakeRotate_;
            if(speedRotateUpDown_ > 0)
                speedRotateUpDown_ -= accelerationRotateUpDown_*dt;
            if(speedRotateUpDown_ < 0)
                speedRotateUpDown_ += accelerationRotateUpDown_*dt;
            if(fabs(speedRotateUpDown_) < accelerationRotateUpDown_*dt)
                speedRotateUpDown_ = 0;
        }

        if(rotateRight_ > 0)
        {
            accelerationRotateRightLeft_ = rotateRight_;
            if(speedRotateRightLeft_ > -maxSpeedRotateRightLeft_)
                speedRotateRightLeft_ -= accelerationRotateRightLeft_*dt;
            if(speedRotateRightLeft_ < -maxSpeedRotateRightLeft_)
                speedRotateRightLeft_ = -maxSpeedRotateRightLeft_;
        }

        if(rotateLeft_ > 0)
        {
            accelerationRotateRightLeft_ = rotateLeft_;
            if(speedRotateRightLeft_ < maxSpeedRotateRightLeft_)
                speedRotateRightLeft_ += accelerationRotateRightLeft_*dt;
            if(speedRotateRightLeft_ > maxSpeedRotateRightLeft_)
                speedRotateRightLeft_ = maxSpeedRotateRightLeft_;
        }

        if(rotateRight_ == 0 && rotateLeft_ == 0)
        {
            accelerationRotateRightLeft_ = brakeRotate_;
            if(speedRotateRightLeft_ > 0)
                speedRotateRightLeft_ -= accelerationRotateRightLeft_*dt;
            if(speedRotateRightLeft_ < 0)
                speedRotateRightLeft_ += accelerationRotateRightLeft_*dt;
            if(fabs(speedRotateRightLeft_) < accelerationRotateRightLeft_*dt)
                speedRotateRightLeft_ = 0;
        }

        if(loopRight_ > 0)
        {
            accelerationLoopRightLeft_ = loopRight_;
            if(speedLoopRightLeft_ < maxSpeedLoopRightLeft_)
                speedLoopRightLeft_ += accelerationLoopRightLeft_*dt;
            if(speedLoopRightLeft_ > maxSpeedLoopRightLeft_)
                speedLoopRightLeft_ = maxSpeedLoopRightLeft_;
        }

        if(loopLeft_ > 0)
        {
            accelerationLoopRightLeft_ = loopLeft_;
            if(speedLoopRightLeft_ > -maxSpeedLoopRightLeft_)
                speedLoopRightLeft_ -= accelerationLoopRightLeft_*dt;
            if(speedLoopRightLeft_ < -maxSpeedLoopRightLeft_)
                speedLoopRightLeft_ = -maxSpeedLoopRightLeft_;
        }

        if(loopLeft_ == 0 && loopRight_ == 0)
        {
            accelerationLoopRightLeft_ = brakeLoop_;
            if(speedLoopRightLeft_ > 0)
                speedLoopRightLeft_ -= accelerationLoopRightLeft_*dt;
            if(speedLoopRightLeft_ < 0)
                speedLoopRightLeft_ += accelerationLoopRightLeft_*dt;
            if(fabs(speedLoopRightLeft_) < accelerationLoopRightLeft_*dt)
                speedLoopRightLeft_ = 0;
        }

        Vector3 transVector = Vector3::ZERO;
*/
/*
        transVector.z = 1;
        this->translate(transVector*speedForward_*dt, Ogre::Node::TS_LOCAL);
        this->pitch(Degree(speedRotateUpDown_*dt), Ogre::Node::TS_LOCAL);
        this->yaw(Degree(speedRotateRightLeft_*dt), Ogre::Node::TS_LOCAL);
        this->roll(Degree(speedLoopRightLeft_*dt), Ogre::Node::TS_LOCAL);
*/
/*
        transVector.x = 1;
        this->translate(transVector*speedForward_*dt, Ogre::Node::TS_LOCAL);
        this->yaw(Degree(speedRotateUpDown_*dt), Ogre::Node::TS_LOCAL);
        this->roll(Degree(speedRotateRightLeft_*dt), Ogre::Node::TS_LOCAL);
        this->pitch(Degree(speedLoopRightLeft_*dt), Ogre::Node::TS_LOCAL);
*/
    }
/*
    void SpaceShip::moveForward(float moveForward) {
        moveForward_ = moveForward;
    }

    void SpaceShip::rotateUp(float rotateUp) {
        rotateUp_ = rotateUp;
    }

    void SpaceShip::rotateDown(float rotateDown) {
        rotateDown_ = rotateDown;
    }

    void SpaceShip::rotateLeft(float rotateLeft) {
        rotateLeft_ = rotateLeft;
    }

    void SpaceShip::rotateRight(float rotateRight) {
        rotateRight_ = rotateRight;
    }

    void SpaceShip::loopLeft(float loopLeft) {
        loopLeft_ = loopLeft;
    }

    void SpaceShip::loopRight(float loopRight) {
        loopRight_ = loopRight;
    }

    void SpaceShip::brakeForward(float brakeForward) {
        brakeForward_ = brakeForward;
    }

    void SpaceShip::brakeRotate(float brakeRotate) {
        brakeRotate_ = brakeRotate;
    }

    void SpaceShip::brakeLoop(float brakeLoop) {
        brakeLoop_ = brakeLoop;
    }

    void SpaceShip::maxSpeedForward(float maxSpeedForward) {
        maxSpeedForward_ = maxSpeedForward;
    }

    void SpaceShip::maxSpeedRotateUpDown(float maxSpeedRotateUpDown) {
        maxSpeedRotateUpDown_ = maxSpeedRotateUpDown;
    }

    void SpaceShip::maxSpeedRotateRightLeft(float maxSpeedRotateRightLeft) {
        maxSpeedRotateRightLeft_ = maxSpeedRotateRightLeft;
    }

    void SpaceShip::maxSpeedLoopRightLeft(float maxSpeedLoopRightLeft) {
        maxSpeedLoopRightLeft_ = maxSpeedLoopRightLeft;
    }
*/
}
