#include "mouseapiexample.h"

namespace orxonox
{

RegisterClass(MouseAPIExample);

MouseAPIExample::MouseAPIExample(Context* context) : ControllableEntity(context)
{
    RegisterObject(MouseAPIExample);
}

MouseAPIExample::~MouseAPIExample()
{
    if(MouseAPI::isActive())
        MouseAPI::getInstance().deactivate();
}

// change the size of the cube by a random number between 0.5 and 5 by clicking on it
void MouseAPIExample::changesizeonclick(MouseButtonCode::ByEnum mouse)
{
    // generate random number between 0.5 and 5
    float randomnumber = (rand()%10+1)/2.0;
    // scale of the cube with this random number
    this->setScale(randomnumber);
    // change the radius of the clickableObject to the new size
    MouseAPI::getInstance().changeRadiusOfClickableObject(cubeid,randomnumber*10);
}

// change the size of the sphere by scrolling on it
void MouseAPIExample::changesizeonscroll(int abs,int rel,const IntVector2& mousePos)
{
    // get current radius of the sphere
    float curRadius = MouseAPI::getInstance().getRadiusScroll(sphereid);
    // set factor to 120% or 80% of the current size, depending on increase or decrease
    float factor = curRadius/10*(1+rel/600.0);
    // return if factor is outside of range [0.5,5] (limit size)
    if(factor > 5 || factor < 0.5) return;
    //scale the sphere with this factor and change the radius
    this->setScale(factor);
    MouseAPI::getInstance().changeRadiusOfScrollableElement(sphereid,factor*10);
}

// scale the z-component up / down with left-mouse / right-mouse (max. factor 2, min. factor 1)
void MouseAPIExample::clickleft(MouseButtonCode::ByEnum mouse)
{
     // action after left-click
     if (mouse == MouseButtonCode::Left)
     {
         Vector3 scale = this->getScale3D();
         if (scale.z <= 2) this->setScale3D(scale.x,scale.y,scale.z+0.1);
     }
     // action after right-click
     else if (mouse == MouseButtonCode::Right)
     {
         Vector3 scale = this->getScale3D();
         if (scale.z > 1) this->setScale3D(scale.x,scale.y,scale.z-0.1);
     }
}

// scale the y-component up / down with left-mouse / right-mouse (max. factor 2, min. factor 1)
void MouseAPIExample::clickright(MouseButtonCode::ByEnum mouse)
{
    // action after left-click
    if (mouse == MouseButtonCode::Left)
    {
        Vector3 scale = this->getScale3D();
        if (scale.y <= 2) this->setScale3D(scale.x,scale.y+0.1,scale.z);
    }
    // action after right-click
    else if (mouse == MouseButtonCode::Right)
    {
        Vector3 scale = this->getScale3D();
        if (scale.y > 1) this->setScale3D(scale.x,scale.y-0.1,scale.z);
    }
}

// standard XML-Port
void MouseAPIExample::XMLPort(Element& xmlelement, XMLPort::Mode mode)
{
    SUPER(MouseAPIExample, XMLPort, xmlelement, mode);

    // differentiate between several objects by an identifier "id"
    XMLPortParam(MouseAPIExample, "id", setId, getId, xmlelement, mode);
    if(this->getId() == 1) // id == 1; cube
    {
        // add the cube to the list with clickable Objects, set the radius to 10, define the function changesizeonclick to be called after a left-click
        cubeid = MouseAPI::getInstance().addClickableObject(this->getWorldPosition(),10,std::list<MouseButtonCode::ByEnum>{MouseButtonCode::Left},[this](MouseButtonCode::ByEnum mouse){this->changesizeonclick(mouse);});
    }
    else if(this->getId() == 2) // id == 2; sphere
    {
        // add the sphere to the list with scrollable Objects, set the radius to 10, define the function changesizeonscroll to be called while scrolling
        sphereid = MouseAPI::getInstance().addScrollElement(this->getWorldPosition(), 10, [this](int abs, int rel, const IntVector2& mousePos){this->changesizeonscroll(abs,rel,mousePos);});
    }
    else if(this->getId() == 3) // id == 3; long block
    {
        // add the left and right part of the long block to the list with clickable Objects and define clickleft/clickright to be called
        leftid = MouseAPI::getInstance().addClickableObject(Vector3(0,70,-70),20,std::list<MouseButtonCode::ByEnum>{MouseButtonCode::Left,MouseButtonCode::Right},[this](MouseButtonCode::ByEnum mouse){this->clickleft(mouse);});
        rightid = MouseAPI::getInstance().addClickableObject(Vector3(0,70,70),20,std::list<MouseButtonCode::ByEnum>{MouseButtonCode::Left,MouseButtonCode::Right},[this](MouseButtonCode::ByEnum mouse){this->clickright(mouse);});
    }

    // activate MouseAPI
    MouseAPI::getInstance().activate();
}
}
