/*
 *   ORXONOX - the hottest 3D action shooter ever to exist
 *                    > www.orxonox.net <
 *
 *
 *   License notice:
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation; either version 2
 *   of the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 *   Author:
 *      Fabian 'x3n' Landau
 *   Co-authors:
 *      ...
 *
 */

/**
    @file OrxoBlox.h
    @brief Declaration of the OrxoBlox class.
    @ingroup OrxoBlox
*/

#ifndef _OrxoBlox_H__
#define _OrxoBlox_H__

#include "OrxoBlox/OrxoBloxPrereqs.h"

#include "tools/Timer.h"

#include "gametypes/Deathmatch.h"
#include "OrxoBloxCenterpoint.h"
#include "OrxoBloxWall.h"

namespace orxonox
{

    /**
    @brief
        Implements a OrxoBlox minigame (<a href="http://en.wikipedia.org/wiki/OrxoBlox">Wikipedia::OrxoBlox</a>).
        It connects the different entities present in a game of OrxoBlox.

        - The @ref orxonox::OrxoBloxCenterpoint "OrxoBloxCenterpoint" is the playing field for the OrxoBlox minigame, it allows for configuration of the minigame, e.g. by setting the size of the playing field, or the length of the @ref orxonox::OrxoBloxBat "OrxoBloxBats". The playing field is always in the x,y-plane, the x-axis being the horizontal and the z-axis being the vertical axis.<br />
        The OrxoBlox class redistributes the important parameters defined in @ref orxonox::OrxoBloxCenterpoint "OrxoBloxCenterpoint" to the other entities, that need to know them, e.g. the @ref orxonox::OrxoBloxBall "OrxoBloxBall" and the @ref orxonox::OrxoBloxBat "OrxoBloxBats".<br />
        The @ref orxonox::OrxoBloxCenterpoint "OrxoBloxCenterpoint" needs to exist in a level with the @ref orxonox::Gametype "Gametype" <em>OrxoBlox</em>.
        - The @ref orxonox::OrxoBloxBall "OrxoBloxBall" is the ball both players play with. The @ref orxonox::OrxoBloxBall "OrxoBloxBall" both implements the movement of the ball, as well as the influence of the boundaries and consequently, also the bouncing (off the upper and lower delimiters, and as off the @ref orxonox::OrxoBloxBat "OrxoBloxBats") of the ball and the effects of the failure of a player to catch the ball (i.e. the scoring of the other player).
        - The two @ref orxonox::OrxoBloxBat "OrxoBloxBats" are the entities through which the players can actively participate in the game, by controlling them. The @ref orxonox::OrxoBloxBat "OrxoBloxBat" class manages the movement (and restrictions thereof) and the influence of the players on the bats.

    @author
        Fabian 'x3n' Landau

    @ingroup OrxoBlox
    */
    class _OrxoBloxExport OrxoBlox : public Deathmatch
    {
        public:
            OrxoBlox(Context* context); //!< Constructor. Registers and initializes the object.
            virtual ~OrxoBlox(); //!< Destructor. Cleans up, if initialized.

            virtual void start() override; //!< Starts the OrxoBlox minigame.
            virtual void end() override; ///!< Ends the OrxoBlox minigame.

            virtual void spawnPlayer(PlayerInfo* player) override; //!< Spawns the input player.

            OrxoBloxShip* getPlayer();

            /**
            @brief Set the OrxoBloxCenterpoint (the playing field).
            @param center A pointer to the OrxoBloxCenterpoint to be set.
            */
            void setCenterpoint(OrxoBloxCenterpoint* center)
                { this->center_ = center; }

            OrxoBloxCenterpoint* getCenterpoint(void)
                { return this->center_; }

        protected:
        private:
            void startWall(void);
            void createWall(void);
            void startBall(); //!< Starts the ball with some default speed.
            void cleanup(); //!< Cleans up the Gametype by destroying the ball and the bats.

            WeakPtr<OrxoBloxCenterpoint> center_; //!< The playing field.
            WeakPtr<OrxoBloxBall> ball_; //!< The OrxoBlox ball.
            
            PlayerInfo* player_;

            Timer starttimer_; //!< A timer to delay the start of the game.
            
            WeakPtr<OrxoBloxWall> futureWall_;
            WeakPtr<OrxoBloxWall> activeWalls_[6]; //6 is the number of wallls in the game.
    };
}

#endif /* _OrxoBlox_H__ */
