/*
 *   ORXONOX - the hottest 3D action shooter ever to exist
 *                    > www.orxonox.net <
 *
 *
 *   License notice:
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation; either version 2
 *   of the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 *   Author:
 *      Fabian 'x3n' Landau
 *   Co-authors:
 *      ...
 *
 */

/**
    @file OrxoBlox.cc
    @brief Implementation of the OrxoBlox class.
*/

#include "OrxoBlox.h"
#include "Highscore.h"

#include "core/CoreIncludes.h"
#include "core/EventIncludes.h"
#include "core/command/Executor.h"


#include "core/config/ConfigValueIncludes.h"//Remove??

#include "gamestates/GSLevel.h"


#include "chat/ChatManager.h"//Remove?

#include "OrxoBloxCenterpoint.h"
#include "OrxoBloxBall.h"
#include "OrxoBloxBot.h"//Remove??
#include "OrxoBloxStones.h"
#include "OrxoBloxWall.h"
#include "OrxoBloxShip.h"


namespace orxonox
{
    

    RegisterUnloadableClass(OrxoBlox);

    /**
    @brief
        Constructor. Registers and initializes the object.
    */
    OrxoBlox::OrxoBlox(Context* context) : Deathmatch(context)
    {
        RegisterObject(OrxoBlox);

        this->center_ = nullptr;
        this->ball_ = nullptr;
        this->futureWall_ = nullptr;
        this->player_ = nullptr;
        

        this->setHUDTemplate("OrxoBloxHUD");
        //Error when specified

        // Pre-set the timer, but don't start it yet.
        this->starttimer_.setTimer(1.0, false, createExecutor(createFunctor(&OrxoBlox::startBall, this)));
        this->starttimer_.stopTimer();

        

    }

    /**
    @brief
        Destructor. Cleans up, if initialized.
    */
    OrxoBlox::~OrxoBlox()
    {
        if (this->isInitialized())
            this->cleanup();
    }

    /**
    @brief
        Cleans up the Gametype by destroying the ball and the bats.
    */
    void OrxoBlox::cleanup()
    {
        if (this->ball_ != nullptr) // Destroy the ball, if present.
        {
            this->ball_->destroy();
            this->ball_ = nullptr;
        }

        if (this->futureWall_)
            {
                this->futureWall_->destroy();
                this->futureWall_ = nullptr;
            }

        // Destroy 6 bWalls, if present.
        for (size_t i = 0; i < 6; ++i)
        {
            if (this->activeWalls_[0] != nullptr)
            {
                this->activeWalls_[0]->destroy();
                this->activeWalls_[0] = nullptr;
            }
            
        }

    }

    /**
    @brieftt    
        Starts the OrxoBlox minigame.
    */
    void OrxoBlox::start()
    {
        if (this->center_ != nullptr) // There needs to be a OrxoBloxCenterpoint, i.e. the area the game takes place.
        {
            if (this->ball_ == nullptr) // If there is no ball, create a new ball.
            {
                this->ball_ = new OrxoBloxBall(this->center_->getContext());
                // Apply the template for the ball specified by the centerpoint.
                this->ball_->addTemplate(this->center_->getBalltemplate());
            }

            // Attach the ball to the centerpoint and set the parameters as specified in the centerpoint, the ball is attached to.
            this->center_->attach(this->ball_);
            this->ball_->setPosition(0, 0, 50);
            this->ball_->setFieldDimension(this->center_->getFieldDimension());
            this->ball_->setSpeed(0);
            this->ball_->setAccelerationFactor(this->center_->getBallAccelerationFactor());
            this->ball_->setBatLength(this->center_->getBatLength());


            // Create the first Wall.
            this->createWall();

        }
        else // If no centerpoint was specified, an error is thrown and the level is exited.
        {
            orxout(internal_error) << "OrxoBlox: No Centerpoint specified." << endl;
            GSLevel::startMainMenu();
            return;
        }

        // Start the timer. After it has expired the ball is started.
        this->starttimer_.startTimer();

        // Set variable to temporarily force the player to spawn.
        bool temp = this->bForceSpawn_;
        this->bForceSpawn_ = true;

        // Call start for the parent class.
        Deathmatch::start();

        // Reset the variable.
        this->bForceSpawn_ = temp;
    }

    /**
    @brief
        Ends the OrxoBlox minigame.
    */
    void OrxoBlox::end()
    {
        ChatManager::message("You suck!!");
        this->cleanup();

        // Call end for the parent class.
        Deathmatch::end();
    }

    /**
    @brief
        Spawns the input player.
    @param player
        The player to be spawned.
    */
    void OrxoBlox::spawnPlayer(PlayerInfo* player)
    {
         assert(player);

        if(this->player_ == nullptr)
        {
            this->player_ = player;
            this->players_[player].state_ = PlayerState::Alive;
        }
    }

    OrxoBloxShip* OrxoBlox::getPlayer()
    {
        for (OrxoBloxShip* ship : ObjectList<OrxoBloxShip>())
        {
            return ship;
        }
        return nullptr;
    }

    //void startWall(void);
    


    void OrxoBlox::createWall(void){
        this->futureWall_ = new OrxoBloxWall(this->center_->getContext());
        // Apply the stone template to the stone.
        this->futureWall_->addTemplate(this->center_->getWallTemplate());

        // Attach the brick to the Centerpoint and set the position of the brick to be at the left side.
        this->center_->attach(this->futureWall_);
        
        
        this->futureWall_->setPosition(10, 10, 0.0f);
        this->futureWall_->setGame(this);
    }


    /**
    @brief
        Starts the ball with some default speed.
    */
    void OrxoBlox::startBall()
    {
        if (this->ball_ != nullptr && this->center_ != nullptr)
            this->ball_->setSpeed(this->center_->getBallSpeed());
    }

    
    
}
