/*
 *   ORXONOX - the hottest 3D action shooter ever to exist
 *                    > www.orxonox.net <
 *
 *
 *   License notice:
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation; either version 2
 *   of the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 *   Author:
 *      Fabian 'x3n' Landau
 *   Co-authors:
 *      ...
 *
 */

/**
    @file OrxoBloxBall.h
    @brief Declaration of the OrxoBloxBall class.
    @ingroup OrxoBlox
*/

#ifndef _OrxoBloxBall_H__
#define _OrxoBloxBall_H__

#include "OrxoBlox/OrxoBloxPrereqs.h"

#include "util/Math.h"

#include "worldentities/MovableEntity.h"


namespace orxonox
{

    /**
    @brief
        This class manages the ball for @ref orxonox::OrxoBlox "OrxoBlox".

        It is responsible for both the movement of the ball in the x,y-plane as well as its interaction with the boundaries of the playing field (defined by the @ref orxonox::OrxoBloxCenterpoint "OrxoBloxCenterpoint") and the @ref orxonox::OrxoBloxBlocks "OrxoBloxBlocks". Or more precisely, it makes the ball bounce off then left and right delimiters of the playing field, it makes the ball bounce off the blocks, damages them and also detects when it reaches the lower bound and takes appropriate measures.

    @author
        Fabian 'x3n' Landau

    @ingroup OrxoBlox
    */
    class _OrxoBloxExport OrxoBloxBall : public MovableEntity
    {
        public:
            OrxoBloxBall(Context* context);
            virtual ~OrxoBloxBall();

            virtual void tick(float dt) override;

            virtual void XMLPort(Element& xmlelement, XMLPort::Mode mode) override;

            /**
            @brief Set the dimensions of the playing field.
            @param width The width of the playing field.
            @param height The height of the playing field.
            */
            void setFieldDimension(float width, float height)
                { this->fieldWidth_ = width; this->fieldHeight_ = height; }
            /**
            @brief Get the dimensions of the playing field.
            @param dimension A vector with the width as the first and height as the second 		    component.
            */
            void setFieldDimension(const Vector2& dimension)
                { this->setFieldDimension(dimension.x, dimension.y); }
            /**
            @brief Get the dimensions of the playing field.
            @return Returns a vector with the width as the first and height as the second 	  	      component.
            */
            Vector2 getFieldDimension() const
                { return Vector2(this->fieldWidth_, this->fieldHeight_); }

            void setSpeed(float speed); //!< Set the speed of the ball (in x-direction).
            /**
            @brief Get the speed of the ball (in x-direction).
            @return Returns the speed of the ball (in x-direction).
            */
            float getSpeed() const
                { return this->speed_; }

            /**
            @brief Set the acceleration factor of the ball.
            @param factor The factor the acceleration of the ball is set to.
            */
            void setAccelerationFactor(float factor)
                { this->accelerationFactor_ = factor; }
            /**
            @brief Get the acceleration factor of the ball.
            @return Returns the acceleration factor of the ball.
            */
            float getAccelerationFactor() const
                { return this->accelerationFactor_; }

            /**
            @brief Set the length of the blocks.
            @param batlength The length of the blocks (in x-direction) as percentage of the height 		    of the playing field.
            */
            void setBlockLength(float blocklength)
                { this->blocklength_ = blocklength; }
            /**
            @brief Get the length of the blocks.
            @return Returns the length of the blocks as percentage of the height of the playing 	    field.
            */
            float getBlockLength() const
		{ return this->blocklength_; }

            void setBlock(WeakPtr<OrxoBloxBlock>* block); //!< Set the blocks for the ball.
            void applyBlock(); //!< Get the block over the network.

            static const float MAX_REL_Z_VELOCITY;

            void setDefScoreSound(const std::string& engineSound);
            const std::string& getDefScoreSound();
            void setDefBatSound(const std::string& engineSound);
            const std::string& getDefBatSound();
            void setDefBoundarySound(const std::string& engineSound);
            const std::string& getDefBoundarySound();

        private:
            void registerVariables();

            float fieldWidth_; //!< The width of the playing field.
            float fieldHeight_; //!< The height of the playing field.
            float speedX_; //!< The speed (in x-direction) of the ball.
            float speedY_; //!< The speed (in y-direction) of the ball.
            float accelerationFactor_; //!< The acceleration factor of the ball.
            float blocklength_; //!< The length of the bats (in z-direction) as percentage of the 	    height of the playing field.
            WeakPtr<OrxoBloxBlock>* block_; //!< An array with the blocks.
            bool bDeleteBlock_; //!< Bool, to keep track, of whether this->block_ exists or not.
            unsigned int* blockID_; //!< The object IDs of the blocks, to be able to synchronize 		    them over the network.
            WorldSound* defScoreSound_;
            WorldSound* defBatSound_;
            WorldSound* defBoundarySound_;
    };
}

#endif /* _OrxoBloxBall_H__ */
