
//TODO: Sounds (all the sounds are still from the pong module...)
//TODO: Blocks (the Ball-Block comunication is based on how the blocks are implemented)
//TODO: The bottom boundary/ the Ball collecter
//TODO: Ability to shoot the ball (the ball is still constructed like the pong ball)

/*
 *   ORXONOX - the hottest 3D action shooter ever to exist
 *                    > www.orxonox.net <
 *
 *
 *   License notice:
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation; either version 2
 *   of the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 *   Author:
 *      Fabian 'x3n' Landau
 *   Co-authors:
 *      ...
 *
 */

/**
    @file OrxoBloxBall.cc
    @brief Implementation of the OrxoBloxBall class.
*/

#include "OrxoBloxBall.h"

#include "core/CoreIncludes.h"
#include "core/GameMode.h"

#include "gametypes/Gametype.h"

#include "OrxoBloxBlocks.h"

#include "sound/WorldSound.h"
#include "core/XMLPort.h"

namespace orxonox
{
    RegisterClass(OrxoBloxBall);

    const float OrxoBloxBall::MAX_REL_Z_VELOCITY = 1.5;

    /**
    @brief
        Constructor. Registers and initializes the object.
    */
    OrxoBloxBall::OrxoBloxBall(Context* context)
        : MovableEntity(context)
    {
        RegisterObject(OrxoBloxBall);

        this->speed_ = 0;
        this->accelerationFactor_ = 1.0f;
        this->block_ = nullptr;
        this->bDeleteBlock_ = false;
        this->blockID_ = new unsigned int[100];
	for (int i = 0; i < 100; i++) {
		this->blockID_[i] = OBJECTID_UNKNOWN;
	}

        this->registerVariables();

        //initialize sound
        if (GameMode::isMaster())
             {
                 this->defScoreSound_ = new WorldSound(this->getContext());
                 this->defScoreSound_->setVolume(1.0f);
                 this->defBatSound_ = new WorldSound(this->getContext());
                 this->defBatSound_->setVolume(0.4f);
                 this->defBoundarySound_ = new WorldSound(this->getContext());
                 this->defBoundarySound_->setVolume(0.5f);
             }
             else
             {
                 this->defScoreSound_ = nullptr;
                 this->defBatSound_ = nullptr;
                 this->defBoundarySound_ = nullptr;
             }
    }

    /**
    @brief
        Destructor.
    */
    OrxoBloxBall::~OrxoBloxBall()
    {
        if (this->isInitialized())
        {
            if (this->bDeleteBlock_)
                delete[] this->block_;

            delete[] this->blockID_;
        }
    }

    //xml port for loading sounds
    void OrxoBloxBall::XMLPort(Element& xmlelement, XMLPort::Mode mode)
    {
        SUPER(OrxoBloxBall, XMLPort, xmlelement, mode);
        XMLPortParam(OrxoBloxBall, "defScoreSound",  setDefScoreSound,  getDefScoreSound,  xmlelement, mode);
        XMLPortParam(OrxoBloxBall, "defBatSound",  setDefBatSound,  getDefBatSound,  xmlelement, mode);
        XMLPortParam(OrxoBloxBall, "defBoundarySound",  setDefBoundarySound,  getDefBoundarySound,  xmlelement, mode);
    }

    /**
    @brief
        Register variables to synchronize over the network.
    */
    void OrxoBloxBall::registerVariables()
    {
        registerVariable( this->fieldWidth_ );
        registerVariable( this->fieldHeight_ );
        registerVariable( this->blocklength_ );
        registerVariable( this->speed_ );
        registerVariable( this->blockID_[0] );
        registerVariable( this->blockID_[1], VariableDirection::ToClient, new NetworkCallback<OrxoBloxBall>( this, &OrxoBloxBall::applyBlock) );
    }

    /**
    @brief
        Is called every tick.
        Handles the movement of the ball and its interaction with the boundaries and blocks.
    @param dt
        The time since the last tick.
    */
    void OrxoBloxBall::tick(float dt)
    {
        SUPER(OrxoBloxBall, tick, dt);

        // Get the current position, velocity and acceleration of the ball.
        Vector3 position = this->getPosition();
        Vector3 velocity = this->getVelocity();
        Vector3 acceleration = this->getAcceleration();

        // If the ball has hit the boundaries on either the right side or the left 
        if (position.x > this->fieldWidth_ / 2 || position.x < -this->fieldWidth_ / 2)
        {
            defBoundarySound_->play(); //play boundary sound
            // Its velocity in x-direction is inverted (i.e. it bounces off).
            velocity.x = -velocity.x;
            // And its position is set as to not overstep the boundary it has just crossed.
            if (position.x > this->fieldWidth_ / 2)
                position.x = this->fieldWidth_ / 2;
            if (position.x < -this->fieldWidth_ / 2)
                position.x = -this->fieldWidth_ / 2;

            this->fireEvent();
        }

	// If the ball has hit the boundary on the top
        if (position.z > this->fieldHeight_ / 2)
        {
            defBoundarySound_->play(); //play boundary sound
            // Its velocity in z-direction is inverted (i.e. it bounces off).
            velocity.z = -velocity.z;
            // And its position is set as to not overstep the boundary it has just crossed.
            position.z = this->fieldHeight_ / 2;

            this->fireEvent();
        }

        // If the ball has crossed the bottom boundary
        if (position.z < -this->fieldHeight_ / 2)
        {
	//TODO: Ball Collector
        }

        // Set the position, velocity and acceleration of the ball, if they have changed.
        if (acceleration != this->getAcceleration())
            this->setAcceleration(acceleration);
        if (velocity != this->getVelocity())
            this->setVelocity(velocity);
        if (position != this->getPosition())
            this->setPosition(position);
    }

    /**
    @brief
        Set the speed of the ball (in x-direction).
    @param speed
        The speed to be set.
    */
    void OrxoBloxBall::setSpeed(float speed)
    {
        if (speed != this->speed_) // If the speed changes
        {
            this->speed_ = speed;

            // Set the speed in the direction of the balls current velocity.
            Vector3 velocity = this->getVelocity();
            if (velocity.x != 0)
                velocity.x = sgn(velocity.x) * this->speed_;
            else // If the balls current velocity is zero, the speed is set in a random direction.
                velocity.x = this->speed_ * sgn(rnd(-1,1));

            this->setVelocity(velocity);
        }
    }

    /**
    @brief
        Set the blocks for the ball.
    @param bats
        An array (of size n (n=#Blocks) of weak pointers, to be set as the new blocks.
    */
    void OrxoBloxBall::setBlock(WeakPtr<OrxoBloxBlock>* block, int n)
    {
        if (this->bDeleteBlock_) // If there are already some blocks, delete them.
        {
            delete[] this->block_;
            this->bDeleteBlock_ = false;
        }

        this->block_ = block;
        // Also store their object IDs, for synchronization.
	for (int i = 0; i < n; i++) {
        	this->blockID_[i] = this->block_[i]->getObjectID();
	}
    }

    /**
    @brief
        Get the blocks over the network.
    */
    void OrxoBloxBall::applyBlock(int n)
    {
        // Make space for the blocks, if they don't exist, yet.
        if (this->block_ == nullptr)
        {
            this->block_ = new WeakPtr<OrxoBloxBlock>[n];
            this->bDeleteBlock_ = true;
        }
	
	for (int i = 0; i < n; i++) {
		if (this->blockID_[i] != OBJECTID_UNKNOWN)
            	this->bat_[i] = orxonox_cast<OrxoBloxBlock*>(Synchronisable::getSynchronisable(this->blockID_[i]));
	}
    }

    void OrxoBloxBall::setDefScoreSound(const std::string &pongSound)
    {
        if( defScoreSound_ )
            defScoreSound_->setSource(pongSound);
        else
            assert(0); // This should never happen, because soundpointer is only available on master
    }

    const std::string& OrxoBloxBall::getDefScoreSound()
    {
        if( defScoreSound_ )
            return defScoreSound_->getSource();
        else
            assert(0);
        return BLANKSTRING;
    }

    void OrxoBloxBall::setDefBatSound(const std::string &pongSound)
    {
        if( defBatSound_ )
            defBatSound_->setSource(pongSound);
        else
            assert(0); // This should never happen, because soundpointer is only available on master
    }

    const std::string& OrxoBloxBall::getDefBatSound()
    {
        if( defBatSound_ )
            return defBatSound_->getSource();
        else
            assert(0);
        return BLANKSTRING;
    }

    void OrxoBloxBall::setDefBoundarySound(const std::string &pongSound)
    {
        if( defBoundarySound_ )
            defBoundarySound_->setSource(pongSound);
        else
            assert(0); // This should never happen, because soundpointer is only available on master
    }

    const std::string& OrxoBloxBall::getDefBoundarySound()
    {
        if( defBoundarySound_ )
            return defBoundarySound_->getSource();
        else
            assert(0);
        return BLANKSTRING;
    }
}
