/*
 *   ORXONOX - the hottest 3D action shooter ever to exist
 *                    > www.orxonox.net <
 *
 *
 *   License notice:
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation; either version 2
 *   of the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 *   Author:
 *      FLeo Merholz
 *      Pascal Schärli
 *   Co-authors:
 *      ...
 *
 */

/**
    @file FlappyOrxShip.cc
    @brief Implementation of the FlappyOrxShip class.
*/

#include "FlappyOrxShip.h"
#include "core/CoreIncludes.h"
#include "core/XMLPort.h"
#include "graphics/Camera.h"
#include "graphics/ParticleSpawner.h"
#include "worldentities/pawns/SpaceShip.h"
#include <math.h> 
#include <ctime>

namespace orxonox
{
    RegisterClass(FlappyOrxShip);

    FlappyOrxShip::FlappyOrxShip(Context* context) : SpaceShip(context)
    {
        RegisterObject(FlappyOrxShip);

        isFlapping = false;
        isDead = true;
        velocity = 0;
        speed = 0;
        upwardThrust = 0;
        gravity = 0;
        particleLifespan = 0.1;
        particleAge = 0;
        deathTime = 0;

        particlespawner_ = NULL;
        
    }

    void FlappyOrxShip::XMLPort(Element& xmlelement, XMLPort::Mode mode)
    {   
        SUPER(FlappyOrxShip, XMLPort, xmlelement, mode);
        XMLPortParam(FlappyOrxShip,"speedBase", setSpeedBase, getSpeedBase, xmlelement,mode);
        XMLPortParam(FlappyOrxShip,"speedIncrease", setSpeedIncrease, getSpeedIncrease, xmlelement,mode);
        XMLPortParam(FlappyOrxShip,"tubeDistanceBase", setTubeDistanceBase, getTubeDistanceBase, xmlelement,mode);
        XMLPortParam(FlappyOrxShip,"tubeDistanceIncrease", setTubeDistanceIncrease, getTubeDistanceIncrease, xmlelement,mode);

        XMLPortParam(FlappyOrxShip,"upwardThrust", setUpwardthrust, getUpwardthrust, xmlelement,mode);
        XMLPortParam(FlappyOrxShip,"gravity", setGravity, getGravity, xmlelement,mode);
    }

    void FlappyOrxShip::tick(float dt)
    {
        SUPER(FlappyOrxShip, tick, dt);


        particleAge+=dt;
        //the particle spawner that generates the fire from the backpack when pressed
        if (particlespawner_ == NULL) {
            for (WorldEntity* object : this->getAttachedObjects())
            {
                if (object->isA(Class(ParticleSpawner)))
                particlespawner_ = (ParticleSpawner*) object;
            }
        }
        else if(particleAge>particleLifespan){
            particlespawner_->setLifetime(1);
        }


        //Execute movement
        if (this->hasLocalController())
        {
            if(getHealth()<0){
                setHealth(1);
                getGame()->death();
                death();   
            }
            Vector3 pos = getPosition();

            getGame()->updatePlayerPos(pos.x);
            

            if(not isDead){
                velocity += gravity*dt;
                if(isFlapping){
                    isFlapping = false;
                    if(pos.z > -150)
                        velocity = -upwardThrust;
                }

                pos += Vector3(speed, 0, velocity) * dt;
            }
            
                        
            // Camera
            Camera* camera = this->getCamera();
            if (camera != nullptr)
            {
                camera->setPosition(pos.x,-100,0);
                camera->setOrientation(Vector3::UNIT_Z, Degree(0));

            }

            
            setPosition(pos);
            setOrientation(Vector3::UNIT_Y, Degree(270-velocity/10));

            if(pos.z > 150){
                getGame()->death();
                death();
            }
            
        }
    }

    time_t FlappyOrxShip::timeUntilRespawn(){
        return 2-time(0)+deathTime;
    }

    void FlappyOrxShip::boost(bool boost){

        if(not isDead){
            particleAge = 0;
        }  
    
        if(isDead&&timeUntilRespawn()<=0){
            isDead = false;
            getGame()->setDead(false);
        }
        isFlapping=boost;
    }

  
    FlappyOrx* FlappyOrxShip::getGame()
    {
        if (game == nullptr)
        {
            for (FlappyOrx* flappyOrx : ObjectList<FlappyOrx>())
                game = flappyOrx;
        }
        return game;
    }

    void FlappyOrxShip::death()
    {
        isDead = true;

        deathTime = time(0);
        Vector3 pos = getPosition();
        pos.x = 0;
        pos.z = 0;
        pos.y = 0;
        velocity = 0;
        setPosition(pos);
    }
}
