/*
 *   ORXONOX - the hottest 3D action shooter ever to exist
 *                    > www.orxonox.net <
 *
 *
 *   License notice:
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation; either version 2
 *   of the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 *   Author:
 *      Fabian 'x3n' Landau
 *   Co-authors:
 *      ...
 *
 */

#include "Mesh.h"

#include <cassert>
#include <sstream>
#include <string>
#include <OgreEntity.h>
#include <OgreSubEntity.h>
#include <OgreSceneManager.h>
#include <OgreHardwareVertexBuffer.h>

#include "util/Convert.h"
#include "core/GameMode.h"

namespace orxonox
{
    unsigned int Mesh::meshCounter_s = 0;

    Mesh::Mesh()
    {
        this->entity_ = nullptr;
        this->bCastShadows_ = true;
    }

    Mesh::~Mesh()
    {
        if (this->entity_ && this->scenemanager_)
            this->scenemanager_->destroyEntity(this->entity_);
    }

    void Mesh::setMeshSource(Ogre::SceneManager* scenemanager, const std::string& meshsource)
    {
        assert(scenemanager);

        this->scenemanager_ = scenemanager;

        if (this->entity_)
            this->scenemanager_->destroyEntity(this->entity_);


        if (GameMode::showsGraphics())
        {
            try
            {
                this->entity_ = this->scenemanager_->createEntity("Mesh" + multi_cast<std::string>(Mesh::meshCounter_s++), meshsource);
                this->entity_->setCastShadows(this->bCastShadows_);


                unsigned short src, dest;
                if (!this->entity_->getMesh()->suggestTangentVectorBuildParams(Ogre::VertexElementSemantic::VES_TANGENT, src, dest))
                {
                    orxout() << "Generate Tanget for \"" << meshsource << '"' << endl;
                    this->entity_->getMesh()->buildTangentVectors(Ogre::VertexElementSemantic::VES_TANGENT, src, dest);
                    // this version cleans mirrored and rotated UVs but requires quality models
                    // mesh->buildTangentVectors(VES_TANGENT, src, dest, true, true);
                }

            }
            catch (...)
            {
                orxout(internal_error) << "Couldn't load mesh \"" << meshsource << '"' << endl;
                this->entity_ = nullptr;
            }
        }
    }

    void Mesh::setCastShadows(bool bCastShadows)
    {
        this->bCastShadows_ = bCastShadows;
        if (this->entity_)
            this->entity_->setCastShadows(this->bCastShadows_);
    }

    const std::string& Mesh::getName() const
    {
        if (this->entity_)
            return this->entity_->getName();
        else
            return BLANKSTRING;
    }

    void Mesh::setMaterial(const std::string& name)
    {
        if (this->entity_)
            this->entity_->setMaterialName(name);
    }

    // PRE: a valid  Ogre::Entity* entity with a valid subentity at index
    // POST: changed material of subentity at index to name
    void Mesh::setSubMaterial(const std::string& name, const int index){
         if (this->entity_ && this->entity_->getSubEntity(index))
        {
                this->entity_->getSubEntity(index)->setMaterialName(name);
        }
        else
            orxout() << "ERROR at Index " << index << endl;
    }

    const std::string& Mesh::getMaterial() const
    {
        if (this->entity_)
        {
            if(this->entity_->getSubEntity(0))
                return this->entity_->getSubEntity(0)->getMaterialName(); //return the Material of the first SubEntity, should be enough for now
            else
                return BLANKSTRING;
        }
        else
            return BLANKSTRING;
    }

    void Mesh::setVisible(bool bVisible)
    {
        if (this->entity_)
            this->entity_->setVisible(bVisible);
    }

    bool Mesh::isVisible() const
    {
        if (this->entity_)
            return this->entity_->getVisible();
        else
            return false;
    }
}
