#ifndef SCRIPTABLE_CONTROLLER_H
#define SCRIPTABLE_CONTROLLER_H

#include <lua.hpp>
#include <string>
#include <list>
#include <map>
#include <memory>
#include "scriptable_controller_api.h"
#include "core/CoreIncludes.h"
#include "worldentities/WorldEntity.h"
#include "worldentities/ControllableEntity.h"
#include "tools/Timer.h"


struct lua_State;

namespace orxonox
{

/**
 * @brief Runs a scripts on a per-level basis and handles the connection to orxonox
 *
 * The script is an attribute of the <Level> element with the name 'script' and should be
 * the path to a lua script. The script will be run as soon as the player spawns. It can
 * then register on various events and react to them. See ScriptableControllerAPI for
 * the complete API.
 *
 * \sa ScriptableControllerAPI
 */
class ScriptableController
{
public:
    /**
     * @brief Run a lua script
     * @param file_path Path to the script
     * @return A lua error code (0 for success)
     *
     * Constructs an API for the script and runs it.
     */
    int runScript(const std::string &file_path);

    /**
     * @brief Set the player object of the current game
     * @param player The player
     *
     * The player is a special object and can perfom special actions.
     */
    void setPlayer(PlayerInfo *player);

    /**
     * @brief Register a WorldEntity to the ScriptableController
     * @param id The ID of the WorldEntity
     * @param entity The WorldEntity
     *
     * The ScriptableController needs a list of all WorldEntity's so it can
     * convert an ID to a WorldEntity.
     */
    void registerWorldEntity(std::string id, WorldEntity *entity);

    /**
     * @brief Register a MobileEntity to the ScriptableController
     * @param id The ID of the MobileEntity
     * @param entity The MobileEntity
     *
     * The ScriptableController needs a list of all MobileEntity's so it can
     * convert an ID to a MobileEntity.
     */
    void registerMobileEntity(std::string id, MobileEntity *entity);

    /**
     * @brief Register a Pawn to the ScriptableController
     * @param id The ID of the Pawn
     * @param pawn The Pawn
     *
     * The ScriptableController needs a list of all Pawn's in addition to
     * the WorldEntity's, because they have additional actions available.
     */
    void registerPawn(std::string id, Pawn *pawn);

    /**
     * @brief Called when a Pawn is killed
     * @param pawn The Pawn
     *
     * Called by the Pawn itself as soon as it's killed.
     */
    void pawnKilled(Pawn *pawn);

    /**
     * @brief Called when a Pawn is hit
     * @param target The hit Pawn
     * @param source The shooting Pawn
     * @param new_health The new health of the hit Pawn
     * @param new_shield The new shield health of the hit Pawn
     *
     * Called by the Pawn itself as soon as it's hit.
     */
    void pawnHit(Pawn *target, Pawn *source, double new_health, double new_shield);

    /**
     * @brief Convert an ID to a WorldEntity pointer
     * @param id The ID of the WorldEntity
     * @return A pointer to the WorldEntity, nullptr if it's not found
     */
    WorldEntity *getWorldEntityByID(std::string id) const;

    /**
     * @brief Convert an ID to a MobileEntity pointer
     * @param id The ID of the MobileEntity
     * @return A pointer to the MobileEntity, nullptr if it's not found
     */
    MobileEntity *getMobileEntityByID(std::string id) const;

    /**
     * @brief Convert an ID to a Pawt pointer
     * @param id The ID of the Pawn
     * @return A pointer to the Pawn, nullptr if it's not found
     */
    Pawn *getPawnByID(std::string id) const;

private:
    std::list<std::unique_ptr<ScriptableControllerAPI> > apis_;
    PlayerInfo *player_;
    std::map<std::string, WorldEntity*> worldEntities_;
    std::map<std::string, MobileEntity*> mobileEntities_;
    std::map<std::string, Pawn*> pawns_;
    std::map<Pawn*, std::string> pawnsReverse_;
    std::map<std::string, ControllableEntity*> controllabelEntities_;

    /**
     * @brief Prints a human readable error message if a lua error occurs
     * @param lua The lua state where the error occured
     */
    void printLuaError(lua_State *lua);
};

}

#endif // SCRIPTABLE_CONTROLLER_H
