
#include "scriptable_controller_api.h"
#include "luatb.h"
#include "scriptable_controller.h"
#include "tools/Timer.h"
#include "worldentities/pawns/Pawn.h"
#include "infos/Bot.h"
#include "worldentities/pawns/ModularSpaceShip.h"

namespace orxonox
{

    const double ScriptableControllerAPI::periodic_interval = 0.5;

    ScriptableControllerAPI::ScriptableControllerAPI(lua_State *lua, ScriptableController *controller)
    {
        this->lua_ = lua;
        this->controller_ = controller;

        // Haven't found a shorter way yet to write that... We need C++17!
        LuaTB<ScriptableControllerAPI, decltype(&ScriptableControllerAPI::orxPrint)>::registerFunction<&ScriptableControllerAPI::orxPrint>(this, lua, "orxPrint");

        LuaTB<ScriptableControllerAPI, decltype(&ScriptableControllerAPI::registerAfterTimeout)>::registerFunction<&ScriptableControllerAPI::registerAfterTimeout>(this, lua, "registerAfterTimeout");
        LuaTB<ScriptableControllerAPI, decltype(&ScriptableControllerAPI::registerAtNearObject)>::registerFunction<&ScriptableControllerAPI::registerAtNearObject>(this, lua, "registerAtNearObject");
        LuaTB<ScriptableControllerAPI, decltype(&ScriptableControllerAPI::registerAtNearPoint)>::registerFunction<&ScriptableControllerAPI::registerAtNearPoint>(this, lua, "registerAtNearPoint");
        LuaTB<ScriptableControllerAPI, decltype(&ScriptableControllerAPI::registerAtAreaEnter)>::registerFunction<&ScriptableControllerAPI::registerAtAreaEnter>(this, lua, "registerAtAreaEnter");
        LuaTB<ScriptableControllerAPI, decltype(&ScriptableControllerAPI::registerAtAreaLeave)>::registerFunction<&ScriptableControllerAPI::registerAtAreaLeave>(this, lua, "registerAtAreaLeave");
        LuaTB<ScriptableControllerAPI, decltype(&ScriptableControllerAPI::registerAtPawnKilled)>::registerFunction<&ScriptableControllerAPI::registerAtPawnKilled>(this, lua, "registerAtPawnKilled");
        LuaTB<ScriptableControllerAPI, decltype(&ScriptableControllerAPI::registerAtPawnHit)>::registerFunction<&ScriptableControllerAPI::registerAtPawnHit>(this, lua, "registerAtPawnHit");

        LuaTB<ScriptableControllerAPI, decltype(&ScriptableControllerAPI::setPosition)>::registerFunction<&ScriptableControllerAPI::setPosition>(this, lua, "setPosition");
        LuaTB<ScriptableControllerAPI, decltype(&ScriptableControllerAPI::setOrientation)>::registerFunction<&ScriptableControllerAPI::setOrientation>(this, lua, "setOrientation");
        LuaTB<ScriptableControllerAPI, decltype(&ScriptableControllerAPI::setVelocity)>::registerFunction<&ScriptableControllerAPI::setVelocity>(this, lua, "setVelocity");
        LuaTB<ScriptableControllerAPI, decltype(&ScriptableControllerAPI::setAngularVelocity)>::registerFunction<&ScriptableControllerAPI::setAngularVelocity>(this, lua, "setAngularVelocity");

        LuaTB<ScriptableControllerAPI, decltype(&ScriptableControllerAPI::killPawn)>::registerFunction<&ScriptableControllerAPI::killPawn>(this, lua, "killPawn");
        LuaTB<ScriptableControllerAPI, decltype(&ScriptableControllerAPI::spawn)>::registerFunction<&ScriptableControllerAPI::spawn>(this, lua, "spawn");



        LuaTB<ScriptableControllerAPI, decltype(&ScriptableControllerAPI::myTestFunction)>::registerFunction<&ScriptableControllerAPI::myTestFunction>(this, lua, "mytestfunction");
        LuaTB<ScriptableControllerAPI, decltype(&ScriptableControllerAPI::moveControllableEntity)>::registerFunction<&ScriptableControllerAPI::moveControllableEntity>(this, lua, "moveControllableEntity");



        this->periodicTimer.setTimer(ScriptableControllerAPI::periodic_interval, true, createExecutor(createFunctor(&ScriptableControllerAPI::periodic, this)), false);
    }

    ScriptableControllerAPI::~ScriptableControllerAPI()
    {
        lua_close(this->lua_);
    }

    void ScriptableControllerAPI::orxPrint(std::string msg)
    {
        orxout(user_info) << msg << std::endl;
    }

    void ScriptableControllerAPI::registerAfterTimeout(std::function<void (void)> callback, double timeout)
    {
        // Kills itself when the timer fires
        new Timer(timeout, false, callback, true);
    }

    //void ScriptableControllerAPI::registerPeriodically(std::function<void (void)> callback, double timeout)
    //{
    //    // Kills itself when the timer fires
    //    new Timer(timeout, false, callback, true);
    //}








    void ScriptableControllerAPI::registerAtNearObject(std::function<void (std::string, std::string)> callback, std::string id1, std::string id2, double distance)
    {
        WorldEntity *entity1 = this->controller_->getWorldEntityByID(id1);
        WorldEntity *entity2 = this->controller_->getWorldEntityByID(id2);

        if(entity1 != nullptr && entity2 != nullptr)
            this->nearObjectHandlers_.push_front(NearObjectHandler(entity1, entity2, id1, id2, distance, callback));
    }

    void ScriptableControllerAPI::registerAtNearPoint(std::function<void (std::string)> callback, std::string id, double x, double y, double z, double distance)
    {
        WorldEntity *entity = this->controller_->getWorldEntityByID(id);

        if(entity != nullptr)
            this->nearPointHandlers_.push_front(NearPointHandler(entity, id, x, y, z, distance, callback));
    }

    void ScriptableControllerAPI::registerAtAreaEnter(std::function<void (std::string)> callback, std::string id, int x, int y, int z, int dx, int dy, int dz)
    {
        WorldEntity *entity = this->controller_->getWorldEntityByID(id);

        if(entity != nullptr)
            this->areaHandlers_.push_front(AreaHandler(entity, id, x, y, z, dx, dy, dz, true, callback));
    }

    void ScriptableControllerAPI::registerAtAreaLeave(std::function<void (std::string)> callback, std::string id, int x, int y, int z, int dx, int dy, int dz)
    {
        WorldEntity *entity = this->controller_->getWorldEntityByID(id);

        if(entity != nullptr)
            this->areaHandlers_.push_front(AreaHandler(entity, id, x, y, z, dx, dy, dz, false, callback));
    }

    void ScriptableControllerAPI::registerAtPawnKilled(std::function<void (std::string)> callback, std::string id)
    {
        this->pawnDestroyedHandlers_[id].push_back(callback);
    }

    void ScriptableControllerAPI::registerAtPawnHit(std::function<void (std::string, std::string, double, double)> callback, std::string id)
    {
        this->pawnHitHandlers_[id].push_back(callback);
    }

    void ScriptableControllerAPI::killPawn(std::string id)
    {
        Pawn *pawn = this->controller_->getPawnByID(id);
        if(pawn == nullptr)
            orxout(user_warning) << "Trying to kill an unknown pawn" << std::endl;
        else
            pawn->kill();
    }

    void ScriptableControllerAPI::spawn(std::string type, std::string id)
    {
        if(this->controller_->getWorldEntityByID(id) != nullptr)
        {
            orxout(user_warning) << "Script tried to spawn an object, but an object with the given ID exists already" << std::endl;
            return;
        }

        Identifier *identifier = ClassByString(type);
        if(!identifier)
        {
            orxout(user_error) << "Script tried to spawn unknown object" << std::endl;
            return;
        }

        if(!identifier->isLoadable())
        {
            orxout(user_error) << "Script tried to spawn unloadable object" << std::endl;
            return;
        }

        WorldEntity *entity;
        Identifiable *obj = identifier->fabricate(this->controller_->getWorldEntityByID("Player")->getContext());

        if(obj->isA(ClassIdentifier<WorldEntity>::getIdentifier()))
        {
            entity = orxonox_cast<WorldEntity*>(obj);
        }
        else if(obj->isA(ClassIdentifier<PlayerInfo>::getIdentifier()))
        {
            // TODO This does not work yet because somehow the controllable entity is not set
            // yet at this stage.
    //        entity = orxonox_cast<PlayerInfo*>(obj)->getControllableEntity();


            //use TEMPLATES in the map to define objects that are not present on the map yet
            return;
        }
        else
        {
            orxout(user_warning) << "Script tried to spawn an object that is neither a WorldEntity, nor a PlayerInfo" << std::endl;
            
            return;
        }

        if(entity->isA(ClassIdentifier<MobileEntity>::getIdentifier()))
            this->controller_->registerMobileEntity(id, orxonox_cast<MobileEntity*>(entity));

        if(entity->isA(ClassIdentifier<Pawn>::getIdentifier()))
            this->controller_->registerPawn(id, orxonox_cast<Pawn*>(entity));

        this->controller_->registerWorldEntity(id, orxonox_cast<WorldEntity*>(entity));
    }

    void ScriptableControllerAPI::setPosition(std::string id, double x, double y, double z)
    {
        WorldEntity *entity = this->controller_->getWorldEntityByID(id);
        if(entity == nullptr)
        {
            orxout(user_warning) << "Trying to set position of an unknown object" << std::endl;
            return;
        }

        const Vector3 &old = entity->getPosition();

        // If one of the values is NaN, don't change that value
        x = std::isnan(x) ? old.x : x;
        y = std::isnan(y) ? old.y : y;
        z = std::isnan(z) ? old.z : z;

        entity->setPosition(x, y, z);
    }

    void ScriptableControllerAPI::setOrientation(std::string id, double x, double y, double z, double angle)
    {
        WorldEntity *entity = this->controller_->getWorldEntityByID(id);
        if(entity == nullptr)
        {
            orxout(user_warning) << "Trying to set orientation of an unknown object" << std::endl;
            return;
        }

        Vector3 old_axis;
        Degree old_angle;

        entity->getOrientation().ToAngleAxis(old_angle, old_axis);

        // If one of the values is NaN, don't change that value
        x = std::isnan(x) ? old_axis.x : x;
        y = std::isnan(y) ? old_axis.y : y;
        z = std::isnan(z) ? old_axis.z : z;
        angle = std::isnan(x) ? old_angle.valueDegrees() : angle;


        entity->setOrientation(Vector3(x, y, z), Degree(angle));
    }

    void ScriptableControllerAPI::setVelocity(std::string id, double x, double y, double z)
    {
        MobileEntity *entity = this->controller_->getMobileEntityByID(id);
        if(entity == nullptr)
        {
            orxout(user_warning) << "Trying to set velocity of an unknown object" << std::endl;
            return;
        }

        const Vector3 &old = entity->getVelocity();

        // If one of the values is NaN, don't change that value
        x = std::isnan(x) ? old.x : x;
        y = std::isnan(y) ? old.y : y;
        z = std::isnan(z) ? old.z : z;

        entity->setVelocity(x, y, z);
    }

    void ScriptableControllerAPI::setAngularVelocity(std::string id, double x, double y, double z)
    {
        MobileEntity *entity = this->controller_->getMobileEntityByID(id);
        if(entity == nullptr)
        {
            orxout(user_warning) << "Trying to set angular velocity of an unknown object" << std::endl;
            return;
        }

        const Vector3 &old = entity->getAngularVelocity();

        // If one of the values is NaN, don't change that value
        x = std::isnan(x) ? old.x : x;
        y = std::isnan(y) ? old.y : y;
        z = std::isnan(z) ? old.z : z;

        entity->setAngularVelocity(x, y, z);
    }

    void ScriptableControllerAPI::pawnKilled(std::string id, Pawn *pawn)
    {
        for(auto callback : this->pawnDestroyedHandlers_[id])
            callback(id);

        this->pawnDestroyedHandlers_.erase(id);

        // We need to delete those handlers as well, they're no longer valid
        auto near_obj_handler = this->nearObjectHandlers_.begin();
        while(near_obj_handler != this->nearObjectHandlers_.end())
        {
            if(near_obj_handler->entity1_ == pawn || near_obj_handler->entity2_ == pawn)
                near_obj_handler = this->nearObjectHandlers_.erase(near_obj_handler);
            else
                near_obj_handler++;
        }

        auto near_point_handler = this->nearPointHandlers_.begin();
        while(near_point_handler != this->nearPointHandlers_.end())
        {
            if(near_point_handler->entity_ == pawn)
                near_point_handler = this->nearPointHandlers_.erase(near_point_handler);
            else
                near_point_handler++;
        }

        auto area_handler = this->areaHandlers_.begin();
        while(area_handler != this->areaHandlers_.end())
        {
            if(area_handler->entity_ == pawn)
                area_handler = this->areaHandlers_.erase(area_handler);
            else
                area_handler++;
        }
    }

    void ScriptableControllerAPI::pawnHit(std::string target_id, std::string source_id, double new_health, double new_shield)
    {
        for(auto callback : this->pawnHitHandlers_[target_id])
            callback(target_id, source_id, new_health, new_shield);
    }

    void ScriptableControllerAPI::periodic()
    {
        // Near object
        auto near_obj_handler = this->nearObjectHandlers_.begin();
        while(near_obj_handler != this->nearObjectHandlers_.end())
        {
            if((near_obj_handler->entity1_->getPosition() - near_obj_handler->entity2_->getPosition()).length() < near_obj_handler->distance_)
            {
                near_obj_handler->callback_(near_obj_handler->id1_, near_obj_handler->id2_);
                near_obj_handler = this->nearObjectHandlers_.erase(near_obj_handler);
            }
            else
            {
                near_obj_handler++;
            }
        }

        // Near point
        auto near_point_handler = this->nearPointHandlers_.begin();
        while(near_point_handler != this->nearPointHandlers_.end())
        {
            if((near_point_handler->entity_->getPosition() - near_point_handler->point_).length() < near_point_handler->distance_)
            {
                near_point_handler->callback_(near_point_handler->id_);
                near_point_handler = this->nearPointHandlers_.erase(near_point_handler);
            }
            else
            {
                near_point_handler++;
            }
        }

        // Areas
        auto area_handler = this->areaHandlers_.begin();
        while(area_handler != this->areaHandlers_.end())
        {
            if(area_handler->entity_->getPosition() > area_handler->start_point_ &&
               area_handler->entity_->getPosition() < area_handler->end_point_)
            {
                if(area_handler->atEnter_)
                {
                    area_handler->callback_(area_handler->id_);
                    area_handler = this->areaHandlers_.erase(area_handler);
                }
                else
                {
                    area_handler++;
                }
            }
            else
            {
                if(!area_handler->atEnter_)
                {
                    area_handler->callback_(area_handler->id_);
                    area_handler = this->areaHandlers_.erase(area_handler);
                }
                else
                {
                    area_handler++;
                }
            }
        }

    }


    //// TESTTESTTESTTESTTESTTEST
    double ScriptableControllerAPI::myTestFunction(double x, double y)
    {
        double z = x + y;
        orxout(user_info) << "Result = " << z << endl;
        return z;
    }

    void ScriptableControllerAPI::moveControllableEntity(std::string id, double x, double y, double z)
    {
        MobileEntity *entity = this->controller_->getMobileEntityByID(id);
        if(entity == nullptr)
        {
            orxout(user_warning) << "Trying to set velocity of an unknown object" << std::endl;
            return;
        }




        Identifier *identifier = ClassByString("ControllableEntity");
       
        ControllableEntity *controllable_entity;
        
        if(identifier->isA(ClassIdentifier<ControllableEntity>::getIdentifier()))
        {
            controllable_entity = orxonox_cast<ControllableEntity*>(entity);

            controllable_entity->moveFrontBack(x);
            controllable_entity->moveRightLeft(y);
            controllable_entity->moveUpDown(z);
        }

        return;
        
    }
}
