/*
 *   ORXONOX - the hottest 3D action shooter ever to exist
 *                    > www.orxonox.net <
 *
 *
 *   License notice:
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation; either version 2
 *   of the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 *   Author:
 *      Fabian 'x3n' Landau
 *   Co-authors:
 *      ...
 *
 */

#include "Level.h"

#include "util/Math.h"
#include "util/SubString.h"
#include "core/CoreIncludes.h"
#include "core/Loader.h"
#include "core/Template.h"
#include "core/XMLFile.h"
#include "core/XMLPort.h"
#include "core/module/PluginReference.h"

#include "infos/PlayerInfo.h"
#include "gametypes/Gametype.h"
#include "overlays/OverlayGroup.h"
#include "LevelManager.h"
#include "scriptablecontroller/scriptable_controller.h"

namespace orxonox
{
    RegisterClass(Level);

    Level::Level(Context* context) : BaseObject(context), Synchronisable(context), Context(context)
    {
        RegisterObject(Level);

        this->setLevel(WeakPtr<Level>(this)); // store a weak-pointer to itself (a strong-pointer would create a recursive dependency)

        this->registerVariables();
        this->xmlfilename_ = this->getFilename();
        this->xmlfile_ = nullptr;
        this->controller_.reset(new ScriptableController());
    }

    Level::~Level()
    {
        if (this->isInitialized())
        {
            if (LevelManager::exists())
                LevelManager::getInstance().releaseActivity(this);

            if (this->xmlfile_)
                Loader::getInstance().unload(this->xmlfile_);

            this->unloadPlugins();
        }
    }

    void Level::XMLPort(Element& xmlelement, XMLPort::Mode mode)
    {
        SUPER(Level, XMLPort, xmlelement, mode);

        XMLPortParam(Level, "plugins",  setPluginsString,  getPluginsString,  xmlelement, mode);
        XMLPortParam(Level, "gametype", setGametypeString, getGametypeString, xmlelement, mode).defaultValues("Gametype");

        XMLPortParamLoadOnly(Level, "script", setScript, xmlelement, mode).defaultValues("scripts/empty.lua");

        XMLPortObject(Level, MeshLodInformation, "lodinformation", addLodInfo, getLodInfo, xmlelement, mode);
        XMLPortObjectExtended(Level, BaseObject, "", addObject, getObject, xmlelement, mode, true, false);
    }

    void Level::registerVariables()
    {
        registerVariable(this->xmlfilename_,            VariableDirection::ToClient, new NetworkCallback<Level>(this, &Level::networkcallback_applyXMLFile));
        registerVariable(this->name_,                   VariableDirection::ToClient, new NetworkCallback<Level>(this, &Level::changedName));
        registerVariable(this->networkTemplateNames_,   VariableDirection::ToClient, new NetworkCallback<Level>(this, &Level::networkCallbackTemplatesChanged));
    }

    void Level::networkcallback_applyXMLFile()
    {
        orxout(user_status) << "Loading level \"" << this->xmlfilename_ << "\"..." << endl;

        ClassTreeMask mask;
        mask.exclude(Class(BaseObject));
        mask.include(Class(Template));
        mask.include(Class(OverlayGroup)); // HACK to include the ChatOverlay

        this->xmlfile_ = new XMLFile(mask, this->xmlfilename_);

        Loader::getInstance().load(this->xmlfile_);
    }

    void Level::networkCallbackTemplatesChanged()
    {
        for(const std::string& name : this->networkTemplateNames_)
        {
            assert(Template::getTemplate(name));
            Template::getTemplate(name)->applyOn(this);
        }
    }

    void Level::setPluginsString(const std::string& pluginsString)
    {
        // unload old plugins
        this->unloadPlugins();

        // load new plugins
        this->pluginsString_ = pluginsString;
        SubString tokens(pluginsString, ",");
        for (size_t i = 0; i < tokens.size(); ++i)
            this->plugins_.push_back(new PluginReference(tokens[i]));
    }

    void Level::unloadPlugins()
    {
        // use destroyLater() - this ensures that plugins are not unloaded too early.
        // Note: When a level gets unloaded, the Level object is usually the last object that gets destroyed. This is because all other
        //       objects inside a level have a StrongPtr (in BaseObject) that references the Level object. This means that the Level
        //       object is only destroyed, when all StrongPtrs that pointed to it were destroyed. But at the time when the last StrongPtr
        //       is destroyed, the other object is not yet fully destroyed because the StrongPtr is destroyed in ~BaseObject (and this
        //       means that e.g. ~Identifiable was not yet called for this object). This means that technically there are still other
        //       objects alive when ~Level is called. This is the reason why we cannot directly destroy() the Plugins - instead we need
        //       to call destroyLater() to ensure that no instances from this plugin exist anymore.
        for (PluginReference* plugin : this->plugins_)
            plugin->destroyLater();
        this->plugins_.clear();
    }

    void Level::setGametypeString(const std::string& gametype)
    {
        Identifier* identifier = ClassByString(gametype);

        if (!identifier || !identifier->isA(Class(Gametype)))
        {
            orxout(internal_error) << "\"" << gametype << "\" is not a valid gametype." << endl;
            identifier = Class(Gametype);
            this->gametype_ = "Gametype";
        }
        else
            this->gametype_ = gametype;

        Gametype* rootgametype = orxonox_cast<Gametype*>(identifier->fabricate(this));

        // store a weak-pointer to the gametype to avoid a circular dependency between this level and the gametype (which has a strong-reference on this level)
        this->setGametype(WeakPtr<Gametype>(rootgametype));

        rootgametype->init(); // call init() AFTER the gametype was set

        if (LevelManager::exists())
            LevelManager::getInstance().requestActivity(this);
    }


    void Level::addObject(BaseObject* object)
    {
        this->objects_.push_back(object);
    }

    BaseObject* Level::getObject(unsigned int index) const
    {
        unsigned int i = 0;
        for (BaseObject* object : this->objects_)
        {
            if (i == index)
                return object;
            ++i;
        }
        return nullptr;
    }

    void Level::addLodInfo(MeshLodInformation* lodInformation)
    {
        std::string meshName = lodInformation->getMeshName();
//         this->lodInformation_.insert(std::make_pair(meshName,lodInformation));
        if( this->lodInformation_.find(meshName) != this->lodInformation_.end())
            orxout(verbose, context::lod) << "replacing lod information for " << meshName << endl;
        this->lodInformation_[meshName] = lodInformation;
    }

    MeshLodInformation* Level::getLodInfo(std::string meshName) const
    {
        if(this->lodInformation_.find(meshName)!=this->lodInformation_.end())
            return this->lodInformation_.find(meshName)->second;

        return nullptr;
    }

    void Level::playerEntered(PlayerInfo* player)
    {
        orxout(internal_info) << "player entered level (id: " << player->getClientID() << ", name: " << player->getName() << ')' << endl;
        player->switchGametype(this->getGametype());
    }

    void Level::playerLeft(PlayerInfo* player)
    {
        orxout(internal_info) << "player left level (id: " << player->getClientID() << ", name: " << player->getName() << ')' << endl;
        player->switchGametype(nullptr);
    }
}
