/*
 *   ORXONOX - the hottest 3D action shooter ever to exist
 *                    > www.orxonox.net <
 *
 *
 *   License notice:
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation; either version 2
 *   of the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 *   Author:
 *      Oli Scheuss
 *   Co-authors:
 *      Damian 'Mozork' Frick
 *
 */

#include "PacmanGhost.h"

#include "core/CoreIncludes.h"
#include "BulletDynamics/Dynamics/btRigidBody.h"

namespace orxonox
{
    RegisterClass(PacmanGhost);

    /**
    @brief
        Constructor. Registers the object and initializes some default values.
    @param creator
        The creator of this object.
    */
    PacmanGhost::PacmanGhost(Context* context) : ControllableEntity(context)
    {
        RegisterObject(PacmanGhost);

        this->localLinearAcceleration_.setValue(1, 1, 1);
        this->localAngularAcceleration_.setValue(1, 1, 1);

        this->velocity = Vector3(0, 0, 0);

        this->setCollisionType(CollisionType::Dynamic);

        this->resetposition = Vector3(0,20,245); //Set Default start position
        
        this->actuelposition = this->getPosition();
        
        this->target_x = actuelposition.x;
        this->target_z = actuelposition.z; 

    }

    /**
    @brief
        Destructor. Destroys controller, if present.
    */
    PacmanGhost::~PacmanGhost()
    {
        // Deletes the controller if the object was initialized and the pointer to the controller is not NULL.
    }

    /**
    @brief
        Method for creating a AutonomousDrone through XML.
    */
    void PacmanGhost::XMLPort(Element& xmlelement, XMLPort::Mode mode)
    {
        SUPER(PacmanGhost, XMLPort, xmlelement, mode);

        XMLPortParam(PacmanGhost, "primaryThrust", setPrimaryThrust, getPrimaryThrust, xmlelement, mode);
        XMLPortParam(PacmanGhost, "auxiliaryThrust", setAuxiliaryThrust, getAuxiliaryThrust, xmlelement, mode);
        XMLPortParam(PacmanGhost, "rotationThrust", setRotationThrust, getRotationThrust, xmlelement, mode);
        XMLPortParam(PacmanGhost, "resetposition", setResetPosition, getResetPosition, xmlelement, mode);
    }


    
    Vector3 possibleposition[] = {Vector3(0,10,245),Vector3(215,0,240)};

    /**
    @brief
        Defines which actions the AutonomousDrone has to take in each tick.
    @param dt
        The length of the tick.
    */
    void PacmanGhost::tick(float dt)
    {
        SUPER(PacmanGhost, tick, dt);

        this->localLinearAcceleration_.setX(this->localLinearAcceleration_.x() * getMass() * this->auxiliaryThrust_);
        this->localLinearAcceleration_.setY(this->localLinearAcceleration_.y() * getMass() * this->auxiliaryThrust_);
        if (this->localLinearAcceleration_.z() > 0)
            this->localLinearAcceleration_.setZ(this->localLinearAcceleration_.z() * getMass() * this->auxiliaryThrust_);
        else
            this->localLinearAcceleration_.setZ(this->localLinearAcceleration_.z() * getMass() * this->primaryThrust_);
        this->physicalBody_->applyCentralForce(physicalBody_->getWorldTransform().getBasis() * this->localLinearAcceleration_);
        this->localLinearAcceleration_.setValue(0, 0, 0);

        this->localAngularAcceleration_ *= this->getLocalInertia() * this->rotationThrust_;
        this->physicalBody_->applyTorque(physicalBody_->getWorldTransform().getBasis() * this->localAngularAcceleration_);
        this->localAngularAcceleration_.setValue(0, 0, 0);


        this->actuelposition = this->getPosition();
        
        //Stop, if target arrived
        if(((this->actuelposition.x - this->target_x)<0.1) && ((this->actuelposition.z - this->target_z)<0.1)){
                 this->ismoving = false;
        }

        //Move, if ghost hasn't arrived yet
        if(this->ismoving){
            if(!((this->actuelposition.z-target_z)<0.1)) {
                velocity = Vector3(-sgn(this->actuelposition.x-this->target_x),-sgn(this->actuelposition.y),-sgn(this->actuelposition.z-this->target_z));
                move(dt);
            }    
            if(!((this->actuelposition.x-target_x)<0.1)){
                velocity = Vector3(-sgn(this->actuelposition.x-this->target_x),-sgn(this->actuelposition.y-this->target_y),-sgn(this->actuelposition.z-this->target_z));
                move(dt);
            }
        }

        //Check on which position ghost has arrived and set new target
         else{
            if(((this->actuelposition.x - possibleposition[0].x)<0.1) && ((this->actuelposition.z - possibleposition[0].z)<0.1)){
                this->target_x = possibleposition[1].x;
                this->target_z = possibleposition[1].z; 
                this->ismoving = true;
            }
            else if(((actuelposition.x - possibleposition[1].x)<0.1) && ((actuelposition.z - possibleposition[1].z)<0.1)){
                this->target_x = possibleposition[0].x;
                this->target_z = possibleposition[0].z; 
                this->ismoving = true;
            }
            else{
            } //End of Position table
            }

    }



    void move(float dt){

        this->setPosition(Vector3(actuelposition.x+velocity.x*dt,actuelposition.y+velocity.y*dt ,actuelposition.z+velocity.z*dt);
    }


    /**
    @brief
        Moves the AutonomousDrone in the negative z-direction (Front/Back) by an amount specified by the first component of the input 2-dim vector.
    @param value
        The vector determining the amount of the movement.
    */
    void PacmanGhost::moveFrontBack(const Vector2& value)
    {
        //this->setPosition(dt*(actuelposition + velocity_));
    }

    /**
    @brief
        Moves the AutonomousDrone in the x-direction (Right/Left) by an amount specified by the first component of the input 2-dim vector.
    @param value
        The vector determining the amount of the movement.
    */
    void PacmanGhost::moveRightLeft(const Vector2& value)
    {
        this->localLinearAcceleration_.setX(this->localLinearAcceleration_.x() + value.x);
    }

    /**
    @brief
        Moves the AutonomousDrone in the y-direction (Up/Down) by an amount specified by the first component of the input 2-dim vector.
    @param value
        The vector determining the amount of the movement.
    */
    void PacmanGhost::moveUpDown(const Vector2& value)
    {
        this->localLinearAcceleration_.setY(this->localLinearAcceleration_.y() + value.x);
    }

    /**
    @brief
        Rotates the AutonomousDrone around the y-axis by the amount specified by the first component of the input 2-dim vector.
    @param value
        The vector determining the amount of the angular movement.
    */
    void PacmanGhost::rotateYaw(const Vector2& value)
    {
        this->localAngularAcceleration_.setY(this->localAngularAcceleration_.y() - value.x);
    }

    /**
    @brief
        Rotates the AutonomousDrone around the x-axis by the amount specified by the first component of the input 2-dim vector.
    @param value
        The vector determining the amount of the angular movement.
    */
    void PacmanGhost::rotatePitch(const Vector2& value)
    {
        this->localAngularAcceleration_.setX(this->localAngularAcceleration_.x() + value.x);
    }

    /**
    @brief
        Rotates the AutonomousDrone around the z-axis by the amount specified by the first component of the input 2-dim vector.
    @param value
        The vector determining the amount of the angular movement.
    */
    void PacmanGhost::rotateRoll(const Vector2& value)
    {
        this->localAngularAcceleration_.setZ(this->localAngularAcceleration_.z() + value.x);
    }

}
