/*
 *   ORXONOX - the hottest 3D action shooter ever to exist
 *                    > www.orxonox.net <
 *
 *
 *   License notice:
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation; either version 2
 *   of the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 *   
 *   This HUD is used for the implementation of the campaign map.
 *   So far it can only be used to give names to an object planet.
 *
 *   Author:
 *      Nikola Bolt
 *   Co-authors:
 *      Claudio Fanconi
 */


// Header file of this cc file
#include "StoryModeHUD.h"

// Ogre stuff
#include <OgreCamera.h>
#include <OgreFontManager.h>
#include <OgreOverlayManager.h>
#include <OgreTextAreaOverlayElement.h>
#include <OgrePanelOverlayElement.h>

// Other stuff
#include <typeinfo>

#include "util/Convert.h"
#include "core/command/ConsoleCommandIncludes.h"
#include "core/CoreIncludes.h"
#include "core/XMLPort.h"
#include "CameraManager.h"
#include "graphics/Camera.h"
#include "worldentities/pawns/Pawn.h"
#include "worldentities/WorldEntity.h"
#include "core/config/ConfigValueIncludes.h"
#include "tools/TextureGenerator.h"
#include "controllers/NewHumanController.h"

#include "worldentities/NameableStaticEntity.h"

namespace orxonox
{
    RegisterClass ( StoryModeHUD );

    // Constructor of the StoryMode HUD
    StoryModeHUD::StoryModeHUD(Context* context) : OrxonoxOverlay(context)
    {
        RegisterObject(StoryModeHUD);

        this->textSize_ = 1.0;
    }

    // Destructor of the StoryMode HUD
    StoryModeHUD::~StoryModeHUD()
    {
        for(Ogre::TextAreaOverlayElement* text : texts)
            Ogre::OverlayManager::getSingleton().destroyOverlayElement(text);
    }

    // Functions of the StoryMode HUD

    // XML Port for Level construction.
    void StoryModeHUD::XMLPort(Element& xmlelement, XMLPort::Mode mode)
    {
        SUPER(StoryModeHUD, XMLPort, xmlelement, mode);

        XMLPortParam(StoryModeHUD, "font", setFont, getFont, xmlelement, mode);
        XMLPortParam(StoryModeHUD, "textSize", setTextSize, getTextSize, xmlelement, mode);
    }

    void StoryModeHUD::initialize(){
        firstTick = false;
        // Scales used for dimensions and text size
        float xScale = this->getActualSize().x;
        float yScale = this->getActualSize().y;

        //Sets the Camera angle at 30 degrees above it, so levels can be seen better    
        CameraManager::getInstance().getActiveCamera()->setOrientation(Vector3::UNIT_X, Degree(-30));

        for(NameableStaticEntity* planet : ObjectList<NameableStaticEntity>()){


            Ogre::TextAreaOverlayElement* text = static_cast<Ogre::TextAreaOverlayElement*>( Ogre::OverlayManager::getSingleton()
                    .createOverlayElement("TextArea", "StoryModeHUD_navText_" + getUniqueNumberString()));

            texts.push_back(text);
            text->setDimensions(xScale, yScale);

            //font name of the text needs to be set here, not in the xml setter function
            text->setFontName(this->fontName_);
            text->setCharHeight(this->textSize_ * yScale);

            //set text
            text->setCaption(planet->getLevelName());

            text->hide();
        
            this->background_->addChild(text);
        }
    }

    // Set the Font of this HUD.
    void StoryModeHUD::setFont(const std::string& font)
    {
        const Ogre::ResourcePtr& fontPtr = Ogre::FontManager::getSingleton().getByName(font);
        if (fontPtr.isNull())
        {
            this->fontName_ = "Monofur";
            orxout(internal_warning) << "StoryModeHUD: Font '" << font << "' not found. Font has been set to Monofur." << endl;
            return;
        }
        this->fontName_ = font;
    }
   
    // Gets the Font of this HUD    
    const std::string& StoryModeHUD::getFont() const
    {
        return this->fontName_;
    }

    // Set the size of the Text
    void StoryModeHUD::setTextSize(float size)
    {
        if (size <= 0.0f)
        {
            this->textSize_ = 0.05f;
            orxout(internal_warning) << "StoryModeHUD: Non positive font size not allowed. Font size has been set to 0.05" << endl;
            return;
        }
        this->textSize_ = size;
        
    }

    // returns the Size of the Text
    float StoryModeHUD::getTextSize() const
    {
        return this->textSize_;
    }

    // Tick: this is the most important function. It's recalled every frame and makes sure things happen on the screen.
    void StoryModeHUD::tick(float dt)
    {
        SUPER(StoryModeHUD, tick, dt);

        if(firstTick)
            this->initialize();

        // cam is the pointer which represents your camera
        Camera* cam = CameraManager::getInstance().getActiveCamera();
        if (cam == nullptr)
            return;

        // camTransform is a Matrix, which converts 3D world of the game into 2D on your screen
        const Matrix4& camTransform = cam->getOgreCamera()->getProjectionMatrix() * cam->getOgreCamera()->getViewMatrix();

        size_t i = 0;
        for(NameableStaticEntity* planet : ObjectList<NameableStaticEntity>()){
            if (i >= texts.size()) {
                break;
            }
            Ogre::TextAreaOverlayElement* text = texts[i];
            i++;

            // Transform to screen coordinates
            Vector3 pos = camTransform * planet->getWorldPosition();

            // If you fly passed the description, it gets out of sight
            if (!(pos.z > 1.0)){
                    
                // Position text
                text->setLeft((pos.x+1)/2); // The (0,0) Coordinate is in the upper left corner.
                text->setTop((-pos.y+1)/2);  // With those two calculations we set the desired positions

                // Make sure the overlays are shown
                text->show();
            }
        }
    }
}
