/*
 *   ORXONOX - the hottest 3D action shooter ever to exist
 *                    > www.orxonox.net <
 *
 *
 *   License notice:
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation; either version 2
 *   of the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 *   Author:
 *      Patrick Wintermeyer
 *   Co-authors:
 *      ...
 *
 */

#include <vector>
#include <string>

#include <OgreOverlayManager.h>
#include <OgreOverlayElement.h>
#include <OgrePanelOverlayElement.h>

#include "HUDPickupSystem.h"

#include "core/CoreIncludes.h"
#include "core/class/Super.h"
#include "util/Convert.h"
#include "pickup/PickupManager.h"
#include "worldentities/pawns/Pawn.h"

namespace orxonox
{
    RegisterClass(HUDPickupSystem);

    HUDPickupSystem::HUDPickupSystem(Context* context) : OrxonoxOverlay(context)
    {
        RegisterObject(HUDPickupSystem);

        const float offsetX = 0.02f;
        const float offsetY = 0.05f;
        const float x = 0.2;
        const float y = 0.5f;

        for (int index = 0; index < 10; ++index)
        {
            int row = index / 5;
            int column = index % 5;

            std::string name = "HUDPickupItem" + multi_cast<std::string>(index);
            Ogre::OverlayElement* item = Ogre::OverlayManager::getSingleton().createOverlayElement("Panel", name);

            item->setDimensions(0.16, 0.4);
            item->setPosition(offsetX + column*x, offsetY + row*y);
            this->items_.push_back(item);
        }
    }

    HUDPickupSystem::~HUDPickupSystem()
    {
        for (Ogre::OverlayElement* item : items_)
            Ogre::OverlayManager::getSingleton().destroyOverlayElement(item);
    }

    void HUDPickupSystem::changedOwner()
    {
        SUPER(HUDPickupSystem, changedOwner);

        this->owner_ = orxonox_cast<Pawn*>(this->getOwner());
    }

    void HUDPickupSystem::tick(float dt)
    {
        SUPER(HUDPickupSystem, tick, dt);

        std::vector<const PickupInventoryContainer*> containers = this->getPickupsForOwner();

        for (size_t index = 0; index < this->items_.size(); ++index)
        {
            Ogre::OverlayElement* item = this->items_[index];

            if (index < containers.size())
            {
                item->setMaterialName(this->getIcon(containers[index]->representationName));
                if (!item->getParent())
                    this->background_->addChild(item);
            }
            else
            {
                if (item->getParent())
                    this->background_->removeChild(item->getName());
            }
        }
    }

    std::vector<const PickupInventoryContainer*> HUDPickupSystem::getPickupsForOwner() const
    {
        std::vector<const PickupInventoryContainer*> containers;

        int numPickups = PickupManager::getInstance().getNumPickups();
        for (int i = 0; i < numPickups; ++i)
        {
            const PickupInventoryContainer* container = PickupManager::getInstance().popPickup();
            if (this->owner_ && this->owner_->getObjectID() == container->carrierPawnId)
                containers.push_back(container);
        }

        return containers;
    }

    std::string HUDPickupSystem::getIcon(const std::string& repName) const
    {
        if(repName.find("invisible", 0)!=std::string::npos) return "Eye";
        else if(repName.find("tri", 0)!=std::string::npos) return "Asterisk";
        else if(repName.find("health", 0)!=std::string::npos || repName.find("Health", 0)!=std::string::npos) return "Cross";
        else if(repName.find("shield", 0)!=std::string::npos) return "Shield";
        else if(repName.find("munition", 0)!=std::string::npos) return "Munition";
        else if(repName.find("shrink", 0)!=std::string::npos) return "Shrink";
        else if(repName.find("boost", 0)!=std::string::npos) return "Flash";
        else if(repName.find("speed", 0)!=std::string::npos) return "3arrowsup";
        else if(repName.find("drone", 0)!=std::string::npos) return "Damage";
        else return "Unknown";
    }
}
