#ifndef SCRIPTABLE_CONTROLLER_API_H
#define SCRIPTABLE_CONTROLLER_API_H

#include <functional>
#include "core/CoreIncludes.h"
#include "tools/Timer.h"
#include "OgreVector3.h"

struct lua_State;

namespace orxonox
{

class ScriptableController;
class WorldEntity;

class ScriptableControllerAPI
{
public:
    ScriptableControllerAPI(lua_State *lua, ScriptableController *controller);
    ~ScriptableControllerAPI();

    void testOutput(std::function<void(std::string)> callback);

    void orxPrint(std::string msg);
    void registerAfterTimeout(std::function<void (void)> callback, double timeout);

    void registerAtNearObject(std::function<void(std::string, std::string)> callback, std::string id1, std::string id2, double distance);
    void registerAtNearPoint(std::function<void (std::string)> callback, std::string id, double x, double y, double z, double distance);
    void registerAtAreaEnter(std::function<void (std::string)> callback, std::string obj, int x, int y, int z, int dx, int dy, int dz);
    void registerAtAreaLeave(std::function<void (std::string)> callback, std::string obj, int x, int y, int z, int dx, int dy, int dz);

    void registerAtObjectDestroyed(std::function<void (std::string)> callback, std::string obj);
    void registerAtPickup(std::function<void (int)> callback, int pickup_id);

    void destroyObject(std::string obj);
    void removeObject(std::string obj);
    void setObjectPosition(std::string obj, double x, double y, double z);

private:
    struct NearObjectHandler
    {
        NearObjectHandler(WorldEntity *entity1, WorldEntity *entity2, std::string id1, std::string id2, double distance, std::function<void (std::string, std::string)> callback)
            : entity1_(entity1), entity2_(entity2), id1_(id1), id2_(id2), distance_(distance), callback_(callback)
        {}

        WorldEntity *entity1_, *entity2_;
        std::string id1_, id2_;
        double distance_;
        std::function<void (std::string, std::string)> callback_;
    };

    struct NearPointHandler
    {
        NearPointHandler(WorldEntity *entity, std::string id, double x, double y, double z, double distance, std::function<void (std::string)> callback)
            : entity_(entity), id_(id), point_(x, y, z), distance_(distance), callback_(callback)
        {}

        WorldEntity *entity_;
        std::string id_;
        Vector3 point_;
        double distance_;
        std::function<void (std::string)> callback_;
    };

    struct AreaHandler
    {
        AreaHandler(WorldEntity *entity, std::string id, double x, double y, double z, double dx, double dy, double dz, bool atEnter, std::function<void (std::string)> callback)
            : entity_(entity), id_(id), start_point_(x, y, z), atEnter_(atEnter), callback_(callback)
        { this-> end_point_ = this->start_point_ + Vector3(dx, dy, dz); }

        WorldEntity *entity_;
        std::string id_;
        Vector3 start_point_, end_point_;
        bool atEnter_;
        std::function<void (std::string)> callback_;
    };


    lua_State *lua_;
    ScriptableController *controller_;
    std::list<NearObjectHandler> nearObjectHandlers_;
    std::list<NearPointHandler> nearPointHandlers_;
    std::list<AreaHandler> areaHandlers_;
    Timer areaCheckTimer;

    void checkAreas(void);
};

}

#endif // SCRIPTABLE_CONTROLLER_API_H
