/*
 *   ORXONOX - the hottest 3D action shooter ever to exist
 *                    > www.orxonox.net <
 *
 *
 *   License notice:
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation; either version 2
 *   of the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 *   Author:
 *      Julien Kindle
 *   Co-authors:
 *      
 *
 */

/**
    @file SOBGumba.cc
    @brief All items in this minigame inherit from this class. Items can move around like platforms and enemies.
*/

#include "SOBGumba.h"

#include "core/CoreIncludes.h"
#include "core/XMLPort.h"
#include "SOBFigure.h"
#include "util/Output.h"


namespace orxonox
{
    RegisterClass(SOBGumba);

    SOBGumba::SOBGumba(Context* context) : MovableEntity(context)
    {
        RegisterObject(SOBGumba);

        attachedToFigure_ = false;
        setAngularFactor(0.0);
        figure_ = nullptr;
        this->enableCollisionCallback();
        gravityAcceleration_ = 350.0;
        speed_ = 0.0;
        hasCollided_=false;
        goesRight_ = true;
        lastPos_ = getPosition();
        lastPos_.x -= 20;
        changeAllowed_ = true;
        changedOn_ = 0.0;
        
    }

    

    void SOBGumba::XMLPort(Element& xmlelement, XMLPort::Mode mode)
    {
        SUPER(SOBGumba, XMLPort, xmlelement, mode);
        XMLPortParam(SOBGumba, "speed", setSpeed, getSpeed, xmlelement, mode);


    }

    
    bool SOBGumba::collidesAgainst(WorldEntity* otherObject, const btCollisionShape* ownCollisionShape, btManifoldPoint& contactPoint) {

        //Every object with mass -1 does not change the direction of the Gumba. For example the ground floor! The other objects switch the direction of the gumba.
        if (changeAllowed_ && otherObject->getMass() != -1) {
            goesRight_ = !goesRight_;
            changeAllowed_ = false;
        }

        return true;
    }


    void SOBGumba::setFigure(SOBFigure* newFigure)
    {
        figure_ = newFigure;
    }



    void SOBGumba::tick(float dt)
    {
        SUPER(SOBGumba, tick, dt);

        if (!changeAllowed_) {
            changedOn_+= dt;
            // After a collision, we don't listen for collisions for 200ms - that's because one wall can cause several collisions!
            if (changedOn_> 0.200) {
                changeAllowed_ = true;
                changedOn_ = 0.0;

            }
        }
    

        int dir = 1;
        if (!goesRight_)
            dir = -1;

        Vector3 velocity = getVelocity();
        velocity.z -= gravityAcceleration_*dt;
        velocity.x = dir*speed_;
        setVelocity(velocity);

        lastPos_ = getPosition();

    }
}
