/*
 *   ORXONOX - the hottest 3D action shooter ever to exist
 *                    > www.orxonox.net <
 *
 *
 *   License notice:
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation; either version 2
 *   of the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 *   Author:
 *      Julien Kindle
 *   Co-authors:
 *      
 *
 */

/**
    @file SOBFigure.cc
    @brief This class represents your figure when you play the minigame. Here the movement of the figure, activating items, ... are handled.
*/

#include "SOBFigure.h"

#include "core/CoreIncludes.h"
#include "core/XMLPort.h"
#include "graphics/Model.h"
#include "graphics/Camera.h"
#include "graphics/ParticleSpawner.h"

#include "SOBMushroom.h"
#include "SOBGumba.h"
#include "SOB.h"
#include "SOBFlagstone.h"
#include "SOBCastlestone.h"
#include <BulletCollision/NarrowPhaseCollision/btManifoldPoint.h>

namespace orxonox
{
    RegisterClass(SOBFigure);

    SOBFigure::SOBFigure(Context* context) : ControllableEntity(context)
    {
        RegisterObject(SOBFigure);

        // initialize variables
        gravityAcceleration_ = 350.0;

        //Vars for movement of player
        moveUpPressed_ = false;
        moveDownPressed_ = false;
        moveLeftPressed_ = false;
        moveDownPressed_ = false;
        firePressed_ = false;
        collDisZ_ = 0;

        //Times and turning
        timeSinceLastFire_ = 0.0;
        lastSpeed_z = 0.0;
        pitch_ = 0.0;
        timeCounter_ = 0;

        //Properties of player
        gotPowerUp_ = false;
        isColliding_ = true;
        particlespawner_ = NULL;

        //Properties of players life
        predead_ = false;
        dead_ = false;
        lvlEnded_ = false;
        reachedLvlEndState_ = 0;

        
        setAngularFactor(0.0); //Means player doesn't turn on collision, so he doesn't fall over while walking over the ground
        this->enableCollisionCallback(); // Turns on that on every collision function collidesAgainst is executed
    }



    bool SOBFigure::collidesAgainst(WorldEntity* otherObject, const btCollisionShape* ownCollisionShape, btManifoldPoint& contactPoint) {

        //Inform tick fct that player is colliding and tell him how far away the collision point is from player middle point in z dir
        isColliding_ = true;
        collDisZ_ = getPosition().z - contactPoint.getPositionWorldOnB().getZ();


        //Orxocast returns object with casted type if otherObject has that class, and if not a nullptr
        SOBMushroom* mush = orxonox_cast<SOBMushroom*>(otherObject);
        SOBGumba* gumba = orxonox_cast<SOBGumba*>(otherObject);
        SOBFlagstone* flagstone = orxonox_cast<SOBFlagstone*>(otherObject);
        SOBCastlestone* castlestone = orxonox_cast<SOBCastlestone*>(otherObject);

        //Check if otherObject is a powerup
        if (mush != nullptr && !(mush->hasCollided_)) {
            otherObject->destroyLater();
            gotPowerUp_ = true;
            SOB* SOBGame = orxonox_cast<SOB*>(getGametype()); //Get the Gametype
            SOBGame->addMushroom(); // Tell the gametype to increase points
            mush->hasCollided_ = true; // needed because of destroyLater takes some time and player should receive points only once


            // now, change the clothes of the Figure to red
            std::string name = "orxo_material_gross";
            this->changeClothes(name);
            
        }
        
        //Check if otherObject is a Gumba (that walking enemies)

         else if (gumba != nullptr && !(gumba->hasCollided_)) {

            //If player jumps on its head, kill the Gumba, else, kill the player
            if (getVelocity().z >= -20) {
                // If player hasn't a power up, he dies. Else he shrinks and the gumba dies.
                if(!gotPowerUp_){
                    Vector3 vel = getVelocity();
                    vel.y = -80;
                    vel.z = 200;
                    setVelocity(vel);
                    predead_=true; 
                    SOB* SOBGame = orxonox_cast<SOB*>(getGametype());
                    SOBGame->setDone(true);
                } else{
                    gotPowerUp_ = false;
                    
                   
           // now, change the clothes of the Figure to old ones
            std::string name = "orxo_material";
            this->changeClothes(name);
                    gumba->destroyLater();
                    gumba->hasCollided_ = true;
                }

          } else {
            gumba->destroyLater();
            gumba->hasCollided_ = true;
            SOB* SOBGame = orxonox_cast<SOB*>(getGametype());
            SOBGame->addGumba();


        }
    }

    //Purpose is that if player hits the flag, he should walk into the castle at the end of the level. For that we use SOBCastlestone
    if (reachedLvlEndState_ == 0 && flagstone != nullptr && !(flagstone->hasCollided_)) {
        flagstone->hasCollided_ = true;
        reachedLvlEndState_ = 1;
        SOB* SOBGame = orxonox_cast<SOB*>(getGametype());
        SOBGame->setDone(true);
        SOBGame->addPoints(flagstone->getPoints());
        

    }
    if (castlestone != nullptr && !(castlestone->hasCollided_)) {
        castlestone->hasCollided_ = true;
        reachedLvlEndState_++;

    }

    return true;
}


//Self implemented sign function that returns either 1 or -1 (and never 0)
int SOBFigure::sgn(float x) {
    if (x < 0.0) return -1;
    return 1;
}

//For those of you who don't have an idea: the tick function is called about 50 times/sec
void SOBFigure::tick(float dt)
{
    SUPER(SOBFigure, tick, dt);


    bool inputAllowed = true;

    //the particle spawner that generates the fire from the backpack when pressed
    if (particlespawner_ == NULL) {
        for (WorldEntity* object : this->getAttachedObjects())
        {
           if (object->isA(Class(ParticleSpawner)))
            particlespawner_ = object;
        }
    }


    //Behavior on level end - this is like described above for the movement from the player when hit the flag. He moves then into the castle
    if (reachedLvlEndState_ != 0) {
        timeCounter_+= dt;
        inputAllowed = false;
    }
    if (reachedLvlEndState_ == 1 && timeCounter_ >= 1.5) {
        timeCounter_ = 0;
        reachedLvlEndState_ = 2;
    }


    //if input blocked, then cancel every movement operation
    if (!inputAllowed) {
        moveUpPressed_ = false;
        moveDownPressed_ = false;
        moveLeftPressed_ = false;
        moveRightPressed_ = false;
    }

    //set the gravityto standard 350
    if (firePressed_ == false) {
        gravityAcceleration_ = 350.0;

    }

    if (hasLocalController())
    {
        Vector3 velocity = getVelocity();
        Vector3 position = getPosition();

        if (!predead_)
            velocity.y = 0;
        //If player falls in a hole
        if (position.z < -100) {
            dead_ = true;
            SOB* SOBGame = orxonox_cast<SOB*>(getGametype());
            SOBGame->setDone(true);
        }


        if (dead_) {
            velocity.x = 0;
            velocity.z = 0;
            setVelocity(velocity);
            SOB* SOBGame = orxonox_cast<SOB*>(getGametype());
            if (firePressed_)
                SOBGame->restart();
            return;
        }


        int maxvelocity_x = 100;
        int speedAddedPerTick = 5;
        int camMaxOffset = 25;

        timeSinceLastFire_ += dt;
        lastSpeed_z = velocity.z;



        //Handle the rocket fire from the jetpack
        if (velocity.z > 40)
            particlespawner_->setVisible(true); 
        else
            particlespawner_->setVisible(false); 


        //If player hits space and collides against an object under him then jump
        if (inputAllowed && firePressed_ && isColliding_ && (collDisZ_ >= 0 && collDisZ_ <+ 10)) {
            gravityAcceleration_ = 100.0;
            velocity.z = 110; 
        }


        //Left-right movement with acceleration and rotation
        float rot = getOrientation().getRoll().valueDegrees();
        if (moveRightPressed_) {
            if (!(rot < 5.0 && -5.0 < rot))
                setOrientation(Vector3::UNIT_Z, getOrientation().getRoll() - sgn(rot)*dt*Radian(6));

            if (std::abs(velocity.x) < maxvelocity_x) {
                velocity.x += speedAddedPerTick;

            }
        } else if (moveLeftPressed_) {
            if (!(abs(rot) > 175.0 ))
                setOrientation(Vector3::UNIT_Z, getOrientation().getRoll() + sgn(rot)*dt*Radian(6));



            if (std::abs(velocity.x) < maxvelocity_x) {
                velocity.x -= speedAddedPerTick;
            }
        } else {
            velocity.x /= 1.1;
        }


        //Again another EndOfLevel behavior
        if (reachedLvlEndState_ == 1)
            velocity.x = -2;
        if (reachedLvlEndState_ == 2)
            velocity.x = 30;
        if (reachedLvlEndState_ == 3) {
            velocity.x = 0;
            velocity.y = 20;
        }
        if (reachedLvlEndState_ == 4) {
            lvlEnded_ = true;
            dead_ = true;
        }

        //velocity = acc. * time
        velocity.z -= gravityAcceleration_*dt;
        setVelocity(velocity);


        //Camera operation - the camera should always follow the player in a specific region
        Camera* cam = getCamera();
        Vector3 campos = cam->getPosition();

        if (campos.x + camMaxOffset < position.x) {
            campos.x = position.x - camMaxOffset;
            cam->setPosition(campos);
        }
        if (campos.x - camMaxOffset > position.x) {
            campos.x = position.x + camMaxOffset;
            cam->setPosition(campos);
        }




    }



    // Reset key variables
    moveUpPressed_ = false;
    moveDownPressed_ = false;
    moveLeftPressed_ = false;
    moveRightPressed_ = false;

    isColliding_ = false;
    collDisZ_ = 0;

}





//The following functions read the input of the player and then set the bools for the movement
void SOBFigure::moveFrontBack(const Vector2& value)
{
    if (value.x > 0)
    {
        moveUpPressed_ = true;
        moveDownPressed_ = false;
    }
    else
    {
        moveUpPressed_ = false;
        moveDownPressed_ = true;
    }
}

void SOBFigure::moveRightLeft(const Vector2& value)
{
    if (value.x > 0)
    {
        moveLeftPressed_ = false;
        moveRightPressed_ = true;
    }
    else
    {
        moveLeftPressed_ = true;
        moveRightPressed_ = false;
    }
}

void SOBFigure::boost(bool boost)
{
    firePressed_ = boost;
}



// PRE: name is an existing name of a material. Example orxo_material for orxo_material.material in data_extern/materials
// POST: clothes of body of player are changed to name
void SOBFigure::changeClothes(std::string& name){
            std::set<WorldEntity*> attachedObjects = this->getAttachedObjects();
            std::set<WorldEntity*>::iterator it;
            for (it = attachedObjects.begin(); it != attachedObjects.end(); ++it)
            {
                Model* FiguresModel = orxonox_cast<Model*>(*it);
                if (FiguresModel != nullptr)
                {

                    FiguresModel->setSubMaterial(name, 4); // 4 is the body

                }
            }   
}

}