#ifndef LUATB_TYPED_STACK_H
#define LUATB_TYPED_STACK_H

#include <type_traits>
#include <functional>
#include <lua.hpp>
#include "is_callable.h"
#include "core/CoreIncludes.h"

// We need a separate class for this because we need to specialize the functions
// and that's not possible if we didn't specialize the class. And the logical
// separation makes sense as well.
class LuaTBTypedStack
{
public:

    // Get a non-function argument from the lua stack and convert it to type 'T'
    // Pops the value from the stack.
    template<typename T>
    static typename std::enable_if<!IsCallable<T>::value, T>::type getArgument(lua_State *lua)
    {
        T value = LuaTBTypedStack::getFromLuaStack<T>(lua);
        lua_pop(lua, 1);
        return value;
    }

    // Specialization if 'T' is a callable type (std::function). Saves the lua
    // function in the registry and constructs a function to call it again.
    // 'decltype(&T::operator())' represents the function signature of the callable
    // object.
    template<typename T>
    static typename std::enable_if<IsCallable<T>::value, T>::type getArgument(lua_State *lua)
    {
        // Here's the tricky part. Apparently, references in the registry can only
        // be called once. This is why we always have to re-add it to the registry
        // after every call. That means, we need a persistent variable for 'ref' and
        // that's only possible with a pointer in this case.
        int *ref = new int(luaL_ref(lua, LUA_REGISTRYINDEX));
        LuaTBTypedStack::callbackRefs.push_back(std::unique_ptr<int>(ref));

        return GetLuaCallback<decltype(&T::operator())>::value(lua, ref);
    }

private:
    template<typename T> struct GetLuaCallback;

    // Gets a value from the top of the lua stack and returns it with the proper type.
    // Does not pop the value!
    template<typename T> static T getFromLuaStack(lua_State *lua);

    // This class is only valid for std::function types. The type argument will not
    // be a normal function signature because it is the signature of
    // std::function<>::operator(), which is also the reason for the 'const'.
    template<typename Ret, typename... Args>
    struct GetLuaCallback<void (std::function<Ret(Args...)>::*)(Args...) const>
    {
        // Returns a lambda that will push the arguments to the lua stack and call
        // the function afterwards. We can't return 'callLuaFunction' directly,
        // because we need the additional arguments 'lua' and 'ref' which we would
        // like to hide from the user.
        static std::function<void (Args...)> value(lua_State *lua, int *ref)
        {
            return [lua, ref](Args... args){callLuaFunction<Ret, Args...>(lua, ref, args...);};
        }
    };

    // Pushes all arguments to the lua stack and calls a lua function afterwards
    template<typename Ret, typename... Args>
    static void callLuaFunction(lua_State *lua, int *ref, Args... args)
    {
        // Get the value of the callback from the registry and push it to the stack
        lua_rawgeti(lua, LUA_REGISTRYINDEX, *ref);

        // Duplicate it on the stack so we can save it in the registry again after
        // we called the callback.
        lua_pushvalue(lua, 1);

        // We pass one extra argument in case the function has no arguments at all
        pushArgumentsToLuaStack<Args...>(lua, args..., false);
        lua_pcall(lua, sizeof...(args), 0, 0);

        // Release the old reference and save the function in the registry again
        lua_unref(lua, *ref);
        *ref = luaL_ref(lua, LUA_REGISTRYINDEX);
    }

    // This is needed in case the function has no arguments at all. Otherwise, we
    // would have a missing argument for such function. This is also why we pass
    // a dummy argument in 'callLuaFunction', so we have at least one argument.
    template<typename T>
    static void pushArgumentsToLuaStack(lua_State *, T)
    { }

    // Recursively pushes arguments to the lua stack
    template<typename First, typename... Args>
    static void pushArgumentsToLuaStack(lua_State *lua, First first, Args... args)
    {
        pushToLuaStack(lua, first);
        pushArgumentsToLuaStack<Args...>(lua, args...);
    }

    // Pushes a value to the lua stack. Only the specializations are valid.
    template<typename T>
    static void pushToLuaStack(lua_State *lua, T value);

    static std::list<std::unique_ptr<int> > callbackRefs;
};

#endif // LUATB_TYPED_STACK_H
