
#include "scriptable_controller.h"
#include "luatb.h"

namespace orxonox
{

// Used https://csl.name/post/lua-and-cpp/ as a reference
int ScriptableController::runScript(const std::string &file_path)
{
    int ret;

    // Create a lua object
    lua_State *lua = luaL_newstate();
    if(lua == nullptr)
    {
        orxout(internal_warning) << "ScriptableController: Falied to load script " << file_path << std::endl;
        return LUA_ERRMEM;
    }

    // Make standard libraries available in the Lua object
    luaL_openlibs(lua);

    // Register the API
    ScriptableControllerAPI *api = new ScriptableControllerAPI(lua, this);

    // Load the program; this supports both source code and bytecode files.
    if((ret = luaL_loadfile(lua, file_path.c_str())) != 0)
    {
        this->printLuaError(lua);
        delete api;
        return ret;
    }

    // Execute the script
    if((ret = lua_pcall(lua, 0, LUA_MULTRET, 0)) != 0)
    {
        this->printLuaError(lua);
        delete api;
        return ret;
    }

    // Remeber the api
    this->apis_.push_back(std::unique_ptr<ScriptableControllerAPI>(api));

    // Only the caller knows, when the end-of-life of the script is reached,
    // so we give him control over when to destroy it.
    return 0;
}

void ScriptableController::registerWorldEntity(int id, WorldEntity *obj)
{
    this->worldEntities_[id] = obj;
}

void ScriptableController::registerControllableEntity(int id, ControllableEntity *obj)
{
    this->worldEntities_[id] = obj;
}

WorldEntity *ScriptableController::getWorldEntityByID(int id) const
{
    auto obj = this->worldEntities_.find(id);
    return obj != this->worldEntities_.end() ? obj->second : nullptr;
}

ControllableEntity *ScriptableController::getControllableEntityByID(int id) const
{
    auto obj = this->controllabelEntities_.find(id);
    return obj != this->controllabelEntities_.end() ? obj->second : nullptr;
}

void ScriptableController::printLuaError(lua_State *lua)
{
    // The error message is on top of the stack.
    // Fetch it, print it and then pop it off the stack.
    // Yes, this is 'const char*' and not 'std::string' because that's what lua gives us.
    const char* message = lua_tostring(lua, -1);
    std::cout << message << std::endl;
    lua_pop(lua, 1);
}

}
