#ifndef LUATB_TYPED_STACK_H
#define LUATB_TYPED_STACK_H

#include <type_traits>
#include <functional>
#include <lua.hpp>
#include "is_callable.h"

// We need a separate class for this because we need to specialize the functions
// and that's not possible if we didn't specialize the class. And the logical
// separation makes sense as well.
class LuaTBTypedStack
{
public:
    // Get an argument from the lua stack and convert it to type 'T'
    template<typename T>
    static T getArgument(lua_State *lua)
    {
        T value = LuaTBTypedStack::popFromLuaStack<T>(lua);
        lua_pop(lua, 1);
        return value;
    }

private:
    // Gets the top element of the lua stack and converts it to the appropriate
    // type. Only works for certain types, if 'T' is none of these types, you'll
    // get an error.
    template<typename T>
    static typename std::enable_if<!IsCallable<T>::value, T>::type popFromLuaStack(lua_State *lua);

    // Specialization if 'T' is a callable type (std::function). Saves the lua
    // function in the registry and constructs a function to call it again.
    // 'decltype(&T::operator())' represents the function signature of the callable
    // object.
    template<typename T>
    static typename std::enable_if<IsCallable<T>::value, T>::type popFromLuaStack(lua_State *lua)
    {
        return GetLuaCallback<decltype(&T::operator())>::value(lua, luaL_ref(lua, LUA_REGISTRYINDEX));
    }

    // Get a function that will call a lua callback
    template<typename T> struct GetLuaCallback;

    // This class is only valid for std::function types. The type argument will not
    // be a normal function signature because it is the signature of
    // std::function<>::operator(), which is also the reason for the 'const'.
    template<typename Ret, typename... Args>
    struct GetLuaCallback<Ret(std::function<Ret(Args...)>::*)(Args...) const>
    {
        // Returns a lambda that will push the arguments to the lua stack and call
        // the function afterwards. We can't return 'callLuaFunction' directly,
        // because we need the additional arguments 'lua' and 'ref' which we would
        // like to hide from the user.
        static std::function<Ret (Args...)> value(lua_State *lua, int ref)
        {
            return [lua, ref](Args... args){return callLuaFunction<Ret, Args...>(lua, ref, args...);};
        }
    };

    // Pushes all arguments to the lua stack and calls a lua function afterwards
    template<typename Ret, typename... Args>
    static Ret callLuaFunction(lua_State *lua, int ref, Args... args)
    {
        lua_rawgeti(lua, LUA_REGISTRYINDEX, ref);
        // We pass one extra argument in case the function has no arguments at all
        pushArgumentsToLuaStack<Args...>(lua, args..., false);
        lua_pcall(lua, 1, sizeof...(args), 0);

        // TODO
        return Ret();
    }
    // This is needed in case the function has no arguments at all. Otherwise, we
    // would have a missing argument for such function. This is also why we pass
    // a dummy argument in 'callLuaFunction', so we have at least one argument.
    template<typename T>
    static void pushArgumentsToLuaStack(lua_State *, T)
    { }

    // Recursively pushes arguments to the lua stack
    template<typename First, typename... Args>
    static void pushArgumentsToLuaStack(lua_State *lua, First first, Args... args)
    {
        pushToLuaStack(lua, first);
        pushArgumentsToLuaStack<Args...>(lua, args...);
    }

    // Pushes a value to the lua stack. Only the specializations are valid.
    template<typename T>
    static void pushToLuaStack(lua_State *lua, T value);
};

#endif // LUATB_TYPED_STACK_H
